package com.viontech.keliu.configuration.elasticsearch;

import cn.hutool.core.util.ClassUtil;
import com.viontech.keliu.configuration.elasticsearch.annotation.ESPrefix;
import com.viontech.keliu.configuration.elasticsearch.annotation.Template;
import com.viontech.keliu.configuration.elasticsearch.util.ESUtil;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.indices.PutIndexTemplateRequest;
import org.elasticsearch.common.xcontent.XContentType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.elasticsearch.annotations.Document;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.index.MappingBuilder;
import org.springframework.data.elasticsearch.core.index.Settings;
import org.springframework.data.elasticsearch.core.mapping.ElasticsearchPersistentEntity;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 启动后ES索引，模板初始化
 * Author: inggg
 * Date: 2021/10/28 15:05
 */
@Component
@ConditionalOnProperty("spring.elasticsearch.rest.uris")
@Slf4j
public class EsInit implements CommandLineRunner {

    @Autowired
    private ElasticsearchRestTemplate template;
    @Autowired
    private ESClient esClient;

    @Override
    public void run(String... args) throws Exception {
        Set<Class<?>> templates = ClassUtil.scanPackageByAnnotation("com.viontech", Template.class);
        for (Class<?> clazz : templates) {
            Template annotation = clazz.getAnnotation(Template.class);
            ESPrefix prefixAnnotation = clazz.getAnnotation(ESPrefix.class);
            String templateName;
            List<String> patterns = new ArrayList<>(annotation.patterns().length);
            if (null != prefixAnnotation) {
                String prefix = ESUtil.getPrefix(prefixAnnotation);
                templateName = prefix + "-" + annotation.templateName();
                for (String pattern : annotation.patterns()) {
                    patterns.add(prefix + "-" + pattern);
                }
            } else {
                templateName = annotation.templateName();
                for (String pattern : annotation.patterns()) {
                    patterns.add(pattern);
                }
            }
            PutIndexTemplateRequest request = new PutIndexTemplateRequest(templateName);

            MappingBuilder mappingBuilder = new MappingBuilder(template.getElasticsearchConverter());
            String mapping = mappingBuilder.buildPropertyMapping(clazz);
            request.patterns(patterns);
            ElasticsearchPersistentEntity<?> persistentEntity = template.getElasticsearchConverter().getMappingContext().getRequiredPersistentEntity(clazz);
            Settings settings = persistentEntity.getDefaultSettings();
            settings.append("index.max_result_window", ESConfig.MAX_SIZE);

            request.mapping(mapping, XContentType.JSON);
            request.settings(settings);
            AcknowledgedResponse acknowledgedResponse = esClient.indices().putTemplate(request, RequestOptions.DEFAULT);
            if (acknowledgedResponse.isAcknowledged()) {
                log.info("{}模板创建完成", templateName);
            }
        }

        Set<Class<?>> classes = ClassUtil.scanPackageByAnnotation("com.viontech", Document.class);
        for (Class<?> clazz : classes) {
            if (clazz.getAnnotation(Document.class).createIndex()) {
                boolean b = template.indexOps(clazz).putMapping(clazz);
                if (b) {
                    log.info("{}索引更新成功", clazz.getName());
                }
            }
        }

        new Thread(()->{
            while (true){
                try{
                    Thread.sleep(1000 * 15);
                    //long l = System.currentTimeMillis();
                    esClient.info();
                    //log.info("es心跳:{}ms", System.currentTimeMillis() - l);
                }catch (Exception e){
                    log.error("EsQueryHandler error",e);
                }
            }
        }).start();
    }
}
