package com.viontech.keliu.chart.excel;

import cn.hutool.json.JSONUtil;
import com.viontech.keliu.chart.Chart;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.model.PieData;
import com.viontech.keliu.chart.series.Series;
import com.viontech.keliu.chart.series.SeriesType;
import com.viontech.keliu.util.NumberUtil;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.util.CellRangeAddress;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

public class ChartExcel {

    private Chart chart;


    public ChartExcel(Chart chart) {
        this.chart = chart;
    }


    private final static int colsizeN = 630;
    private final static int colsizeM = 1000;

    public InputStream creatExcelInputStream() {
        Map<String, Object> excelData = getExcelData(false);
        return Data2Excel(excelData);
    }

    public InputStream creatExcelInputStreamWithSeries() {
        Map<String, Object> excelData = getExcelData(true);
        return Data2Excel(excelData);
    }


    private Map<String, Object> getExcelData(boolean withSeries) {
        Map<String, Object> excelMap = new HashMap<String, Object>();
        String title = chart.getTitle();
        excelMap.put("title", title);

        List<Object> heads = new LinkedList<Object>();

        Map<Object, Object> rows = new LinkedHashMap<Object, Object>();

        Axis axis = chart.getAxis();
        List<String> axisList = new ArrayList<String>();
        if (axis != null) {
            axisList = axis.getData();
        }


        List<Series> series = chart.getSeries();
        boolean flag = true;
        for (int i = 0; i < series.size(); i++) {
            Series serie = series.get(i);
            String serieName = serie.getName();
            String type = serie.getType();

            if (SeriesType.pie.name().equals(type)) {
                //表头
                if (flag) {
                    heads.add("\\");//表头第一个单元格
                    flag = false;
                }
                if (!heads.contains(serieName))
                    heads.add(serieName);
                List<PieData> data = serie.getData();
                for (int j = 0; j < data.size(); j++) {
                    Object name = data.get(j).getName();
                    List<Object> rowList = (List<Object>) rows.get(name);
                    if (rowList == null) {
                        rowList = new ArrayList<Object>();
                        rowList.add(name);
                    }
                    Object value = data.get(j).getValue();
                    rowList.add(value);
                    rows.put(name, rowList);
                }

            } else if (SeriesType.table.name().equals(type)) {
                List data = serie.getData();
                String name = serie.getName();
                if (withSeries) {
                    List<Object> rowList = (List<Object>) rows.get(name);
                    if (rowList == null) {
                        rowList = new ArrayList<Object>();
                        rowList.add(name);
                    }
                    rowList.addAll(data);
                    rows.put(name, rowList);

                    axisList.add(0, "\\");
                } else {
                    rows.put(name, data);
                }
                if (heads.size() <= 0)
                    //为了解决一个商场多列的导出情况
                    for (String s : axisList) {
                        if (s.contains("[") && s.contains("]")) {
                            String[] split = s.replace("[", "").replace("]", "").replace("'", "").split(",");
                            if (split.length > 2) {
                                for (int i1 = 1; i1 < split.length; i1++) {
                                    heads.add("[" + split[0] + "-" + split[i1] + "]");
                                }
                                continue;
                            }
                            heads.add(s);
                            continue;
                        }
                        heads.add(s);
                    }
                //heads.addAll(axisList);
            } else {
                //表头
                if (flag) {
                    heads.add("\\");//表头第一个单元格
                    flag = false;
                }
                if (!heads.contains(serieName))
                    heads.add(serieName);
                List data = serie.getData();
                Set<String> arrSet = new HashSet<>();
                for (int j = 0; j < data.size(); j++) {
                    String name = axisList.get(j);
                    if (!arrSet.isEmpty() && arrSet.contains(name)) {
                        List<Object> rowList = (List<Object>) rows.get(name + " ");
                        if (rowList == null) {
                            rowList = new ArrayList<Object>();
                            rowList.add(name);
                        }
                        rowList.add(data.get(j));
                        rows.put(name + " ", rowList);
                    } else {
                        List<Object> rowList = (List<Object>) rows.get(name);
                        if (rowList == null) {
                            rowList = new ArrayList<Object>();
                            rowList.add(name);
                        }
                        rowList.add(data.get(j));
                        rows.put(name, rowList);
                        arrSet.add(name);
                    }
                }
            }
        }
        excelMap.put("heads", heads);
        excelMap.put("rows", rows);
        return excelMap;
    }

    public InputStream Data2Excel(Map<String, Object> excelMap) {
        try {
            HSSFWorkbook wb = new HSSFWorkbook();
            HSSFSheet sheet = wb.createSheet();
            /**
             * 设置标题样式
             */
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleStyle.setAlignment(HorizontalAlignment.CENTER);
            HSSFFont font = wb.createFont();
            font.setBold(true);
            font.setFontHeight((short) 400);
            titleStyle.setFont(font);
            HSSFCell titleCell = sheet.createRow(0).createCell(0); // 创建第一行，并在该行创建单元格，设置内容，做为标题行
            /**
             * 获取标题
             */
            String title = (String) excelMap.get("title");
            titleCell.setCellValue(new HSSFRichTextString(title));
            titleCell.setCellStyle(titleStyle);


            HSSFRow headRow = sheet.createRow(1);
            int colSzie = 0;
            /**
             * 设置表头样式
             */
            HSSFCellStyle headStyle = wb.createCellStyle();
            headStyle.setAlignment(HorizontalAlignment.CENTER);
            HSSFFont headFont = wb.createFont();
            headFont.setBold(true);
            headFont.setFontHeight((short) 240);
            headStyle.setFont(headFont);

            List<Object> heads = (List<Object>) excelMap.get("heads");
            List<String> columnTypes = new ArrayList<>(heads.size());
            // 表头做一下处理，对于对象类型的表头，只要name字段
            heads = heads.stream().map(h -> {
                String str = (String) h;
                if (JSONUtil.isJsonObj(str)) {
                    Map map = JSONUtil.toBean(str, Map.class);
                    columnTypes.add(map.getOrDefault("type", "null").toString());
                    return map.getOrDefault("name", str).toString();
                } else {
                    columnTypes.add("null");
                    return str;
                }
            }).collect(Collectors.toList());
            Map<Object, Object> rows = (Map<Object, Object>) excelMap.get("rows");

            for (int i = 0; i < heads.size(); i++) {
                String head = (String) heads.get(i);
                HSSFCell cell = headRow.createCell(i);
                cell.setCellValue(new HSSFRichTextString(head));
                cell.setCellStyle(headStyle);
                colSzie++;
                sheet.autoSizeColumn((short) i);
                sheet.setColumnWidth(i, head.length() * colsizeN + colsizeM);
            }
            HSSFCellStyle cellStyle = wb.createCellStyle();
            cellStyle.setAlignment(HorizontalAlignment.RIGHT);

            CellStyle percentageStyle = wb.createCellStyle();
            percentageStyle.setAlignment(HorizontalAlignment.RIGHT);
            percentageStyle.setDataFormat(wb.createDataFormat().getFormat("0.00%"));
            int i = 0;
            for (Object datas : rows.values()) {
                HSSFRow row = sheet.createRow(i + 2);
                List<Object> rowData = (List<Object>) datas;
                for (int j = 0; j < rowData.size(); j++) {
                    HSSFCell cell = row.createCell(j);

                    Object cellValue = rowData.get(j);
                    // 判空处理，为null的单元格设置值为 --
                    if (cellValue == null) {
                        cell.setCellValue("--");
                        cell.setCellStyle(cellStyle);
                        continue;
                    }
                    boolean isNum = false;//data是否为数值型
                    boolean isInteger = false;//data是否为整数
                    boolean isPercent = false;//data是否为百分数
                    boolean isTime = false; //是否是时间，目前数值是整数的秒，需要转成分:秒的格式
                    if (!"".equals(cellValue)) {

                        switch (columnTypes.get(j)) {
                            case "Integer":
                                isNum = true;
                                isInteger = true;
                                break;
                            case "Double":
                                isNum = true;
                                break;
                            case "percent":
                                isNum = true;
                                isPercent = true;
                                break;
                            case "time":
                                isTime = true;
                                break;
                            default:
                                //判断data是否为数值型
                                isNum = cellValue.toString().matches("^(-?\\d+)(\\.\\d+)?$");
                                //判断data是否为整数（小数部分是否为0）
                                isInteger = cellValue.toString().matches("^[-\\+]?[\\d]*$");
                                //判断data是否为百分数（是否包含“%”）
                                isPercent = cellValue.toString().contains("%");
                                break;
                        }
                    }
                    //如果单元格内容是数值类型，涉及到金钱（金额、本、利），则设置cell的类型为数值型，设置data的类型为数值类型
                    if (isNum && !isPercent) {
                        // 虽然单元格指定了是数字类型，但是有些值会是--，不是数字，直接保持原样
                        if (cellValue.toString().matches("^(-?\\d+)(\\.\\d+)?$")) {
                            cell.setCellValue(Double.parseDouble(cellValue.toString()));
                            if (isInteger) {
                                cellStyle.setDataFormat(HSSFDataFormat.getBuiltinFormat("#,#0"));//数据格式只显示整数
                            } else {
                                cellStyle.setDataFormat(HSSFDataFormat.getBuiltinFormat("#,##0.00"));//保留两位小数点
                            }
                        }else {
                            cell.setCellValue(cellValue.toString());
                        }
                        cell.setCellStyle(cellStyle);

                    } else if (isPercent) {
                        if (cellValue.toString().endsWith("%")) {
                            cellValue = cellValue.toString().replace("%", "");
                        }
                        //百分比的值已经是乘以 100 的数了，在设置为Excel的百分比类型时还要再先除以 100，因为Excel显示百分比时会自动乘以 100 并加上 %
                        // 虽然单元格指定了是数字类型，但是有些值会是--，不是数字，直接保持原样
                        if (cellValue.toString().matches("^(-?\\d+)(\\.\\d+)?$")) {
                            cell.setCellValue(NumberUtil.divide(Double.parseDouble(cellValue.toString()), 100, 4));
                            cell.setCellStyle(percentageStyle);
                        }else {
                            cell.setCellValue(cellValue.toString());
                        }

                    } else if (isTime) {
                        // 转换一下数值
                        int second = Integer.parseInt(cellValue.toString());
                        cell.setCellValue(second / 60 + ":" + String.format("%02d", second % 60));
                        cell.setCellStyle(cellStyle);
                    } else {
                        // 设置单元格内容为字符型
                        cell.setCellValue(String.valueOf(cellValue));
                        cell.setCellStyle(cellStyle);
                    }
                    cellStyle.setDataFormat(HSSFDataFormat.getBuiltinFormat(""));
                }
                i++;
            }

            // 当报表数据为空时，合并单元格，默认合并3个
            if (colSzie > 1) {
                sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, colSzie - 1));
            } else {
                sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, 2));
            }
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            try {
                wb.write(os);
            } catch (IOException e) {
                e.printStackTrace();
            }
            byte[] b = os.toByteArray();
            ByteArrayInputStream in = new ByteArrayInputStream(b);
            return in;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

}


