package com.viontech.keliu.chart;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.series.*;
import gnu.trove.list.array.TIntArrayList;

import java.util.*;

public class Chart<T>{
	/**
	 * 图表名称
	 */
	protected String title;
	/**
	 * 图 表key
	 */
	protected String chartKey;

	/**
	 * x轴信息
	 */
	private Axis xAxis;
	/**
	 * y轴信息
	 */
	private Axis yAxis;

	/**
	 * 数据
	 */
	protected List<Series> series;

	/**
	 *数据Map
	 */
	protected Map<String,Series> seriesMap;

	/**
	 * 默认数据类型
	 */
	protected SeriesType defaultSeriesType;

	/**
	 * 求和计算方式
	 */
	public final static int CALC_TYPE_SUM = 0;

	/**
	 * 均值计算方式
	 */
	public final static int CALC_TYPE_AVG = 1;
	/**
	 * 计算方式列表
	 */
	protected TIntArrayList calcTypes;
	/**
	 * 数据个数列表
	 */


	protected int startIndex;

	protected int endIndex;

	protected Integer pageTotal;


	public Chart(String title,SeriesType defaulType) {
		this.seriesMap = new HashMap<String,Series>();
		this.series = new ArrayList<Series>();
		this.title = title;
		this.defaultSeriesType = defaulType;
	}

	/**
	 * 往chart报表中添加数据
	 * @param seriesName 数据名称
	 * @param value 数据值
	 */
	public boolean putValue(String seriesName,Object coordinate,T value){
		Series series = getOrCreatIfNeed(seriesName,null);
		return series.putValueByCoordinate(coordinate,value);
	}

	public boolean putValue(String seriesName,String stack,Object coordinate,T value){
		Series series = getOrCreatIfNeed(seriesName,stack);
		return series.putValueByCoordinate(coordinate,value);
	}

	/**
	 *
	 * @param seriesName
	 * @param coordinate
	 * @param value
	 * @return
	 */
	public boolean adjustOrPutValue(String seriesName,Object coordinate,T value){
		Series series = getOrCreatIfNeed(seriesName,null);
		return series.adjustOrPutValueByCoordinate(coordinate,value);
	}

	public boolean adjustOrPutValue(Object coordinate,T value){
		for (Map.Entry<String, Series> entry: seriesMap.entrySet()) {
			entry.getValue().adjustOrPutValueByCoordinate(coordinate,value);
		}
		return true;
	}
	/**
	 * 添加或累加数据
	 * @param seriesName
	 * @param stack
	 * @param coordinate
	 * @param value
	 * @return
	 */
	public boolean adjustOrPutValue(String seriesName,String stack,Object coordinate,T value){
		Series series = getOrCreatIfNeed(seriesName,stack);
		return series.adjustOrPutValueByCoordinate(coordinate,value);
	}

	/**
	 * 通过坐标设置计算方式
	 * @param coordinate 坐标
	 * @param calcType 计算方式
	 */
	public void setCalcTypeByCoordinate(Object coordinate,int calcType){
		setCalcTypeByIndex(getIndexByCoordinate(coordinate), calcType);
	}
	/**
	 *  通过索引设置计算法方式
	 * @param index 索引
	 * @param calcType 计算方式
	 */
	public void setCalcTypeByIndex(int index,int calcType){
		if(calcTypes == null){
			calcTypes = new TIntArrayList();

			for (int i = 0; i < (getAxis() == null ? 0 : getAxis().getLength()); i++) {
				calcTypes.add(CALC_TYPE_SUM);
			}
		}
		if (index > calcTypes.size()) {
			for(int i=calcTypes.size();i<index+1;i++){
				calcTypes.add(CALC_TYPE_SUM);
			}
		}
		calcTypes.set(index, calcType);
	}



	/**
	 * 通过坐标获取计算方式
	 * @param coordinate 坐标
	 * @return 计算方式
	 */
	public Integer getCalcTypeByCoordinate(Object coordinate){
		return getCalcTypeByIndex(getIndexByCoordinate(coordinate));
	}

	@SuppressWarnings("unchecked")
	public int getIndexByCoordinate(Object coordinate){
		int index = 0;
		if(xAxis!=null){
			index =  xAxis.getIndexByCoordinate(coordinate);
		}else if(yAxis!=null){
			index = yAxis.getIndexByCoordinate(coordinate);
		}
		if(index<getStartIndex()){
			setStartIndex(index);
		}
		if(index>getEndIndex()){
			setEndIndex(index);
		}
//		if (index == -1) {
//			return 0;
//		}
		return index;
	}
	/**
	 * 通过索引获取计算方式
	 * @param index 索引
	 * @return 计算方式
	 */
	public Integer getCalcTypeByIndex(int index){
		if(calcTypes == null) return CALC_TYPE_SUM;
		if(index>=calcTypes.size()){
			return CALC_TYPE_SUM;
		}
		return calcTypes.get(index);
	}
	/**
	 * 设置所有坐标数据的计算方式
	 * @param calcType 计算方式
	 */
	public void setAllCalcType(Integer calcType){
		if(calcTypes == null){
			calcTypes = new TIntArrayList();
			for (int i = 0; i < (getAxis() == null ? 0 : getAxis().getLength()); i++) {
				calcTypes.add(CALC_TYPE_SUM);
			}
		}
		for (int i = 0; i < calcTypes.size(); i++) {
			calcTypes.set(i, calcType);
		}
	}



	/**
	 * 通过名字获取或创建一个报表数据对象
	 * @param seriesName 名字
	 * @return 报表数据对象
	 */
	public Series getOrCreatIfNeed(String seriesName,String stack){
		Series seriesVal = seriesMap.get(seriesName);
		if(seriesVal == null){
			seriesVal = createSeries(seriesName,stack,null);
		}
		return seriesVal;
	}
	/**
	 * 创建chart报表数据对象
	 * @param seriesName
	 * @return
	 */
	public Series createSeries(String seriesName){
		return this.createSeries(seriesName,null,null);
	}

	/**
	 *
	 * @param seriesName
	 * @return
	 */
	public Series getSeries(String seriesName){
		Series s = seriesMap.get(seriesName);
		if(s == null){
			s = getOrCreatIfNeed(seriesName,null);
		}
		return s;
	}

	/**
	 * @param seriesName 需要移除的series name
	 */
	public void removeSeries(String seriesName){
		Series exist = seriesMap.get(seriesName);
		seriesMap.remove(seriesName);
		series.remove(exist);
	}

	public Series createSeries(String seriesName,SeriesType seriesType){
		return this.createSeries(seriesName, null, seriesType);
	}
	public Series createSeries(String seriesName,String stack,SeriesType seriesType){
		if(seriesType == null){
			seriesType = defaultSeriesType;
		}
		Series result = null;
		switch (seriesType) {
			case bar:
				 result = new Bar(seriesName,stack,this);
				 break;
			case line:
				result = new Line(seriesName, stack, this);
				break;
			case pie:
				result = new Pie(seriesName, this);
				break;
			case table:
				result = new Row(seriesName,this);
				break;
			case scatter:
				result = new Scatter(seriesName, this);
				break;
			case radar:
				result = new Radar(seriesName, this);
				break;
			default:
				break;
		}
		Axis axis = getAxis();
		if(axis != null && axis.getLength() > 0){
			result.putValueByIndex(axis.getData().size()-1, null);
		}
		this.series.add(result);
		seriesMap.put(seriesName, result);
		return result;
	}


	@JsonIgnore
	public  void sort(String seriesName,Comparator comparator){
		Series series = getSeries(seriesName);
		series.sort(comparator,getAxis());
	}

	@JsonIgnore
	public void sort(Comparator<Series> comparator){
		Collections.sort(series,comparator);
	}

	@JsonIgnore
	public void subData(int beginIndex,int endIndex){
		getAxis().subData(beginIndex, endIndex);
		for (Series serie : series) {
			serie.subvalue(beginIndex, endIndex);
		}
	}

	public void removeChartHideCol(){
		Axis axis = getAxis();
		if (axis == null)
			return;

		List<String> axis_hideData = axis.getHideData();//获取指定移除的列
		if (axis_hideData == null || axis_hideData.size() <= 0)
			return;

		List data_raw = axis.getData_raw();
		if ( data_raw == null || data_raw.size() <= 0)
			return;

		for (Series seriesCol : series) {
			seriesCol.getData();//手动开始计算,然后再处理
			List<String> axis_data = new ArrayList<>();
			axis_data.addAll(data_raw);
			for (String hideCol : axis_hideData) {
				int index = axis_data.indexOf(hideCol);
				//System.out.println(axis_data.toString()+"移除series数据，"+seriesCol.getName()+"--"+hideCol+"--"+index);
				//System.out.println("移除前数据"+seriesCol.getData().toString());
				axis_data.remove(hideCol);
				seriesCol.removeValue(index);
				//System.out.println("移除后数据"+seriesCol.getData().toString());
			}
		}
		//移除轴上的数据
		List data = axis.getData();
		if (data == null || data.size() <= 0)
			return;
		for (String hideCol : axis_hideData) {
			data.remove(hideCol);
		}
	}

	public T calcValue(int index,List<T> data_raw){
		return calcValue(null,index, data_raw);
	}

	public T calcValue(String name,int index,List<T> data_raw){
		return data_raw.get(index);
	}


	//-------------------------------getter/setter方法-----------------------------------//
	public List<Series> getSeries() {
		return series;
	}
	public void setSeries(List<Series> series) {
		for (Series s : series) {
			seriesMap.put(s.getName(), s);
		}
		this.series = series;
	}

	@JsonIgnore
	public int getStartIndex() {
		return startIndex;
	}

	public void setStartIndex(int startIndex) {
		this.startIndex = startIndex;
	}

	@JsonIgnore
	public int getEndIndex() {
		return endIndex;
	}

	public void setEndIndex(int endIndex) {
		this.endIndex = endIndex;
	}

	public String getTitle() {
		return title;
	}
	public void setTitle(String title) {
		this.title = title;
	}
	@JsonIgnore
	public Axis getAxis(){
		if(xAxis != null){
			return xAxis;
		}
		if(yAxis != null){
			return yAxis;
		}
		return null;
	}

	public Axis getXAxis() {
		return xAxis;
	}

	public void setXAxis(Axis xAxis) {
		this.xAxis = xAxis;
	}

	public Axis getYAxis() {
		return yAxis;
	}

	public void setYAxis(Axis yAxis) {
		this.yAxis = yAxis;
	}

	public String getChartKey() {
		return chartKey;
	}
	public void setChartKey(String chartKey) {
		this.chartKey = chartKey;
	}

	public Integer getPageTotal() {
		return pageTotal;
	}

	public void setPageTotal(Integer pageTotal) {
		this.pageTotal = pageTotal;
	}
}
