package com.viontech.keliu.redis;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

@Component
public class RedisUtil {
	
	protected static  RedisTemplate redisTemplate;
	
	@Autowired(required = false)
	public void setRedisTemplate(RedisTemplate redisTemplate){
		this.redisTemplate = redisTemplate;
	}
	
	
	public static RedisTemplate getRedisTemplate() {
		return redisTemplate;
	}


	public static boolean isInit(){
		return redisTemplate != null;
	}
	/**
	 * 出异常，重复操作的次数
	 */
	private static  Integer times = 5;
	
	 /**
	 * 用户排序通过注册时间的 权重值
	 * @param date
	 * @return
	 */
	public static double getCreateTimeScore(long date) {
	    return date/ 100000.0;
	}
	
	/**
     * 获取Redis中所有的键的key
     * @return
     */
	public static Set<String> getAllKeys() {
	    return redisTemplate.keys("*");
	}

	
	public static Set<String> getKeys(String keyMatcher) {
	    return redisTemplate.keys(keyMatcher);
	}
	/**
     * 获取所有的普通key-value
     * @return
     */
	public static  Map<String, Object> getAllString() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, Object> map = new HashMap<String, Object>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.STRING) {
	            map.put(k, get(k));
	        }
	    }
	    return map;
	}
	
	 /**
     * 获取所有的Set -key-value
     * @return
     */
	public static  Map<String, Set<Object>> getAllSet() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, Set<Object>> map = new HashMap<String, Set<Object>>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.SET) {
	            map.put(k, getSet(k));
	        }
	    }
	    return map;
	}
	
	/**
     * 获取所有的ZSet正序  -key-value 不获取权重值
     * @return
     */
	public static  Map<String, Set<Object>> getAllZSetRange() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, Set<Object>> map = new HashMap<String, Set<Object>>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.ZSET) {
	            map.put(k, getZSetRange(k));
	        }
	    }
	    return map;
	}
	
	/**
     * 获取所有的ZSet倒序  -key-value 不获取权重值
     * @return
     */
	public static  Map<String, Set<Object>> getAllZSetReverseRange() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, Set<Object>> map = new HashMap<String, Set<Object>>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.ZSET) {
	            map.put(k, getZSetReverseRange(k));
	        }
	    }
	    return map;
	}
	
	 /**
     * 获取所有的List -key-value
     * @return
     */
	public static  Map<String, List<Object>> getAllList() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, List<Object>> map = new HashMap<String, List<Object>>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.LIST) {
	            map.put(k, getList(k));
	        }
	    }
	    return map;
	}
	
	 /**
     * 获取所有的Map -key-value
     * @return
     */
	public static  Map<String, Map<String, Object>> getAllMap() {
	    Set<String> stringSet = getAllKeys();
	    Map<String, Map<String, Object>> map = new HashMap<String, Map<String, Object>>();
	    Iterator<String> iterator = stringSet.iterator();
	    while (iterator.hasNext()) {
	        String k = iterator.next();
	        if (getType(k) == DataType.HASH) {
	            map.put(k, getMap(k));
	        }
	    }
	    return map;
	}
	
	 /**
     * 添加一个list
     * @param key
     * @param objectList
     */
	public static  void addList(String key, List<Object> objectList) {
	    for (Object obj : objectList) {
	        addList(key, obj);
	    }
	}
	
	/**
     * 向list中增加值
     * @param key
     * @param obj
     * @return 返回在list中的下标
     */
	public static  long addList(String key, Object obj) {
		//由原来的右进右出，改为坐进右出
		return redisTemplate.boundListOps(key).leftPush(obj);
	    //return redisTemplate.boundListOps(key).rightPush(obj);
	}

	/**
     * 向list中增加值
     * @param key
     * @param obj
     * @return 返回在list中的下标
     */
	public static  long addList(String key, Object... obj) {
		//由原来的右进右出，改为坐进右出
		return redisTemplate.boundListOps(key).leftPushAll(obj);
	    //return redisTemplate.boundListOps(key).rightPushAll(obj);
	}
	
	/**
    *
    * 输出list
    * @param key List的key
    * @param s 开始下标
    * @param e 结束的下标
    * @return
    */
	public static  List<Object> getList(String key, long s, long e) {
	    return redisTemplate.boundListOps(key).range(s, e);
	}
	
	/**
     * 输出完整的list
     * @param key
     */
	public static  List<Object> getList(String key) {
	    return redisTemplate.boundListOps(key).range(0, getListSize(key));
	}
	
	/**
     * 获取list集合中元素的个数
     * @param key
     * @return
     */
	public static  long getListSize(String key) {
	    return redisTemplate.boundListOps(key).size();
	}
	public static Object leftPopWithTimeOut(String key) {
	    return redisTemplate.boundListOps(key).leftPop(0, TimeUnit.SECONDS);
	}

	public static Object leftPop(String key) {
	    return redisTemplate.boundListOps(key).leftPop();
	}
	public static Object rightPopWithTimeOut(String key) {
	    return redisTemplate.boundListOps(key).rightPop(0, TimeUnit.SECONDS);
	}

	public static Object rightPop(String key) {
	    return redisTemplate.boundListOps(key).rightPop();
	}
	/**
     * 移除list中某值
     * 移除list中 count个value为object的值,并且返回移除的数量,
     * 如果count为0,或者大于list中为value为object数量的总和,
     * 那么移除所有value为object的值,并且返回移除数量
     * @param key
     * @param object
     * @return 返回移除数量
     */
	public static  long removeListValue(String key, Object object) {
	    return redisTemplate.boundListOps(key).remove(0, object);
	}
	
	/**
     * 移除list中某值
     * @param key
     * @param object
     * @return 返回移除数量
     */
	public static  long removeListValue(String key, Object... objects) {
	    long r = 0;
	    for (Object object : objects) {
	        r += removeListValue(key, object);
	    }
	    return r;
	}
	
	/**
     * 批量删除key对应的value
     * @param keys
     */
	public static  void remove(String... key) {
	    if (key != null && key.length > 0) {
	        if (key.length == 1) {
	            remove(key[0]);
	        } else {
	            redisTemplate.delete(CollectionUtils.arrayToList(key));
	        }
	    }
	}
	
	 /**
     * 模糊移除 支持*号等匹配移除
     * @param blears
     */
	public static  void removeBlear(String... blears) {
	    for (String blear : blears) {
	        removeBlear(blear);
	    }
	}
	
	/**
     * 修改key名 如果不存在该key或者没有修改成功返回false
     * @param oldKey
     * @param newKey
     * @return
     */
	public static  Boolean renameIfAbsent(String oldKey, String newKey) {
	    return redisTemplate.renameIfAbsent(oldKey, newKey);
	}
	
	/**
     * 模糊移除 支持*号等匹配移除
     * @param blear
     */
	public static  void removeBlear(String blear) {
	    redisTemplate.delete(redisTemplate.keys(blear));
	}
	
	/**
     * 根据正则表达式来移除key-value
     * @param blears
     */
	public static  void removeByRegular(String... blears) {
	    for (String blear : blears) {
	        removeBlear(blear);
	    }
	}
	
	/**
     * 根据正则表达式来移除key-value
     * @param blears
     */
	public static  void removeByRegular(String blear) {
	    Set<String> stringSet = getAllKeys();
	    for (String s : stringSet) {
	        if (Pattern.compile(blear).matcher(s).matches()) {
	            redisTemplate.delete(s);
	        }
	    }
	}
	
	/**
     * 根据正则表达式来移除 Map中的key-value
     * @param key
     * @param blears
     */
	public static  void removeMapFieldByRegular(String key, String... blears) {
	    for (String blear : blears) {
	        removeMapFieldByRegular(key, blear);
	    }
	}
	
	/**
     * 根据正则表达式来移除 Map中的key-value
     * @param key
     * @param blear
     */
	public static  void removeMapFieldByRegular(String key, String blear) {
	    Map<String, Object> map = getMap(key);
	    Set<String> stringSet = map.keySet();
	    for (String s : stringSet) {
	        if (Pattern.compile(blear).matcher(s).matches()) {
	            redisTemplate.boundHashOps(key).delete(s);
	        }
	    }
	}
	
	/**
     * 移除key 对应的value
     * @param key
     * @param value
     * @return
     */
	public static  Long removeZSetValue(String key, Object... value) {
	    return redisTemplate.boundZSetOps(key).remove(value);
	}
	
	/**
     * 移除key ZSet
     * @param key
     * @return
     */
	public static  void removeZSet(String key) {
	    removeZSetRange(key, 0L, getZSetSize(key));
	}
	
	/**
     *删除，键为K的集合，索引start<=index<=end的元素子集
     * @param key
     * @param start
     * @param end
     * @return
     */
	public static  void removeZSetRange(String key, Long start, Long end) {
	    redisTemplate.boundZSetOps(key).removeRange(start, end);
	}
	
	/**
     * 并集 将key对应的集合和key1对应的集合合并到key2中
     * 如果分数相同的值，都会保留
     * 原来key2的值会被覆盖
     * @param key
     * @param key1
     * @param key2
     */
	public static  void setZSetUnionAndStore(String key,String key1, String key2) {
	    redisTemplate.boundZSetOps(key).unionAndStore(key1,key2);
	}
	
	/**
     * 获取整个有序集合ZSET，正序
     * @param key
     */
	public static  Set<Object> getZSetRange(String key) {
	    return getZSetRange(key, 0, getZSetSize(key));
	}
	
	/**
     * 获取有序集合ZSET
     * 键为K的集合，索引start<=index<=end的元素子集，正序
     * @param key
     * @param s 开始位置
     * @param e 结束位置
     */
	public static  Set<Object> getZSetRange(String key, long s, long e) {
	    return redisTemplate.boundZSetOps(key).range(s, e);
	}
	
	/**
     * 获取整个有序集合ZSET，倒序
     * @param key
     */
	public static  Set<Object> getZSetReverseRange(String key) {
	    return getZSetReverseRange(key, 0, getZSetSize(key));
	}
	
	/**
     * 获取有序集合ZSET
     * 键为K的集合，索引start<=index<=end的元素子集，倒序
     * @param key
     * @param start 开始位置
     * @param end 结束位置
     */
	public static  Set<Object> getZSetReverseRange(String key, long start, long end) {
	    return redisTemplate.boundZSetOps(key).reverseRange(start, end);
	}
	
	/**
     * 通过分数(权值)获取ZSET集合 正序 -从小到大
     * @param key
     * @param start
     * @param end
     * @return
     */
	public static  Set<Object> getZSetRangeByScore(String key, double start, double end) {
	    return redisTemplate.boundZSetOps(key).rangeByScore(start, end);
	}
	
	/**
     * 通过分数(权值)获取ZSET集合 倒序 -从大到小
     * @param key
     * @param start
     * @param end
     * @return
     */
	public static  Set<Object> getZSetReverseRangeByScore(String key, double start, double end) {
	    return redisTemplate.boundZSetOps(key).reverseRangeByScore(start, end);
	}
	
	 /**
     * 键为K的集合，索引start<=index<=end的元素子集
     * 返回泛型接口（包括score和value），正序
     * @param key
     * @param start
     * @param end
     * @return
     */
	public static  Set<ZSetOperations.TypedTuple<Object>> getZSetRangeWithScores(String key, long start, long end) {
	    return redisTemplate.boundZSetOps(key).rangeWithScores(start, end);
	}
	
	/**
     * 键为K的集合，索引start<=index<=end的元素子集
     * 返回泛型接口（包括score和value），倒序
     * @param key
     * @param start
     * @param end
     * @return
     */
	public static  Set<ZSetOperations.TypedTuple<Object>> getZSetReverseRangeWithScores(String key, long start, long end) {
	    return redisTemplate.boundZSetOps(key).reverseRangeWithScores(start, end);
	}
	
	/**
     * 键为K的集合
     * 返回泛型接口（包括score和value），正序
     * @param key
     * @return
     */
	public static  Set<ZSetOperations.TypedTuple<Object>> getZSetRangeWithScores(String key) {
	    return getZSetRangeWithScores(key, 0, getZSetSize(key));
	}
	
	/**
     * 键为K的集合
     * 返回泛型接口（包括score和value），倒序
     * @param key
     * @return
     */
	public static  Set<ZSetOperations.TypedTuple<Object>> getZSetReverseRangeWithScores(String key) {
	    return getZSetReverseRangeWithScores(key, 0, getZSetSize(key));
	}
	
	/**
     * 键为K的集合，sMin<=score<=sMax的元素个数
     * @param key
     * @param sMin
     * @param sMax
     * @return
     */
	public static  long getZSetCountSize(String key, double sMin, double sMax) {
	    return redisTemplate.boundZSetOps(key).count(sMin, sMax);
	}
	
	/**
     * 获取Zset 键为K的集合元素个数
     * @param key
     * @return
     */
	public static  long getZSetSize(String key) {
	    return redisTemplate.boundZSetOps(key).size();
	}
	
	/**
     * 获取键为K的集合，value为obj的元素分数
     * @param key
     * @param value
     * @return
     */
	public static  double getZSetScore(String key, Object value) {
	    return redisTemplate.boundZSetOps(key).score(value);
	}
	
	/**
     * 元素分数增加，delta是增量
     * @param key
     * @param value
     * @param delta
     * @return
     */
	public static  double incrementZSetScore(String key, Object value, double delta) {
	    return redisTemplate.boundZSetOps(key).incrementScore(value, delta);
	}
	
	/**
     * 添加有序集合ZSET
     * 默认按照score升序排列，存储格式K(1)==V(n)，V(1)=S(1)
     * @param key
     * @param score
     * @param value
     * @return
     */
	public static  Boolean addZSet(String key, double score, Object value) {
	    return redisTemplate.boundZSetOps(key).add(value, score);
	}
	
	/**
     * 添加有序集合ZSET
     * @param key
     * @param value
     * @return
     */
	public static  Long addZSet(String key, TreeSet<Object> value) {
	    return redisTemplate.boundZSetOps(key).add(value);
	}
	
	/**
     * 添加有序集合ZSET
     * @param key
     * @param score
     * @param value
     * @return
     */
	public static  Boolean addZSet(String key, double[] score, Object[] value) {
	    if (score.length != value.length) {
	        return false;
	    }
	    for (int i = 0; i < score.length; i++) {
	        if (addZSet(key, score[i], value[i]) == false) {
	            return false;
	        }
	    }
	    return true;
	}
	
	/**
     * 删除缓存
     * 根据key精确匹配删除
     * @param key
     */
	public static void remove(String key) {
	    if (exists(key)) {
	        redisTemplate.delete(key);
	    }
	}
	
	/**
     * 通过分数删除ZSet中的值
     * @param key
     * @param s
     * @param e
     */
	public static  void removeZSetRangeByScore(String key,double s , double e) {
	    redisTemplate.boundZSetOps(key).removeRangeByScore(s,e);
	}
	
	/**
     * 设置Set的过期时间
     * @param key
     * @param time
     * @return
     */
	public static  Boolean setSetExpireTime(String key, Long time) {
	    return redisTemplate.boundSetOps(key).expire(time, TimeUnit.SECONDS);
	}
	
	/**
     * 设置ZSet的过期时间
     * @param key
     * @param time
     * @return
     */
	public static  Boolean setZSetExpireTime(String key, Long time) {
	    return redisTemplate.boundZSetOps(key).expire(time, TimeUnit.SECONDS);
	}
	
	/**
     * 判断缓存中是否有key对应的value
     * @param key
     * @return
     */
	public static  boolean exists(String key) {
	    return redisTemplate.hasKey(key);
	}
	
	/**
     * 读取String缓存 可以是对象
     * @param int key
     * @return
     */
	public static Object get(int key) {
	    return get(String.valueOf(key));
	}
	/**
     * 读取String缓存 可以是对象
     * @param long key
     * @return
     */
	public static  Object get(long key) {
	    return get(String.valueOf(key));
	}
	/**
     * 读取String缓存 可以是对象
     * @param String key
     * @return
     */
	public static Object get(String key) {
	    return redisTemplate.boundValueOps(key).get();
	}
	
	/**
     * 读取String缓存 可以是对象
     * @param key
     * @return
     */
	public static  List<Object> get(String... keys) {
	    List<Object> list = new ArrayList<Object>();
	    for (String key : keys) {
	        list.add(get(key));
	    }
	    return list;
	}
	
	 /**
     * 读取缓存 可以是对象 根据正则表达式匹配
     * @param regKey
     * @return
     */
	public static  List<Object> getByRegular(String regKey) {
	    Set<String> stringSet = getAllKeys();
	    List<Object> objectList = new ArrayList<Object>();
	    for (String s : stringSet) {
	        if (Pattern.compile(regKey).matcher(s).matches() && getType(s) == DataType.STRING) {
	            objectList.add(get(s));
	        }
	    }
	    return objectList;
	}
	
	/**
     * 写入缓存 可以是对象
     * @param long key
     * @param value
     */
	public static  void set(long key, Object value) {
	   set(String.valueOf(key) ,value);
	}
	
	/**
     * 写入缓存 可以是对象
     * @param int key
     * @param value
     */
	public static void set(int key, Object value) {
	    set(String.valueOf(key) ,value);
	 }
	
	/**
     * 写入缓存 可以是对象
     * @param string key
     * @param value
     */
	public static void set(String key, Object value) {
	    redisTemplate.boundValueOps(key).set(value);
	}
	
	/**
     * 写入缓存
     * @param key
     * @param value
     * @param expireTime 过期时间 -单位s
     * @return
     */
	public static  void set(String key, Object value, Long expireTime) {
	    redisTemplate.boundValueOps(key).set(value, expireTime, TimeUnit.SECONDS);
	}
	
	/**
     * 设置一个key的过期时间（单位：秒）
     * @param key
     * @param expireTime
     * @return
     */
	public static  boolean setExpireTime(String key, Long expireTime) {
	    return redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
	}

	public static long getExpireTime(String key) {
		return redisTemplate.getExpire(key);
	}
	/**
     * 获取key的类型
     * @param key
     * @return
     */
	public static  DataType getType(String key) {
	    return redisTemplate.type(key);
	}
	
	/**
     * 删除map中的某个对象
     * @param key   map对应的key
     * @param field map中该对象的key
     */
	public static  void removeMapField(String key, Object... field) {
	    redisTemplate.boundHashOps(key).delete(field);
	}
	
	/**
     * 获取map对象
     * @param key map对应的key
     * @return
     */
	public static  Long getMapSize(String key) {
	    return redisTemplate.boundHashOps(key).size();
	}
	
	 /**
     * 获取map对象
     * @param key map对应的key
     * @return
     */
	public static  Map<String, Object> getMap(String key) {
	    return redisTemplate.boundHashOps(key).entries();
	}
	
	/**
     * 获取map缓存中的某个对象
     * @param key map对应的key
     * @param field map中该对象的key
     * @return
     */
	public static  <T> T getMapField(String key, String field) {
	    return (T) redisTemplate.boundHashOps(key).get(field);
	}
	
	/**
     * 判断map中对应key的key是否存在
     * @param key map对应的key
     * @return
     */
	public static  Boolean hasMapKey(String key, String field) {
	    return redisTemplate.boundHashOps(key).hasKey(field);
	}
	
	/**
     * 获取map对应key的value
     * @param key map对应的key
     * @return
     */
	public static  List<Object> getMapFieldValue(String key) {
	    return redisTemplate.boundHashOps(key).values();
	}
	
	/**
     * 获取map的key
     * @param key map对应的key
     * @return
     */
	public static  Set<Object> getMapFieldKey(String key) {
	    return redisTemplate.boundHashOps(key).keys();
	}
	
	/**
     * 添加map
     * @param key
     * @param map
     */
	public static  void addMap(String key, Map<String, Object> map) {
	    redisTemplate.boundHashOps(key).putAll(map);
	}
	
	/**
     * 向key对应的map中添加缓存对象
     * @param key   cache对象key
     * @param field map对应的key
     * @param value     值
     */
	public static  void addMap(String key, String field, Object value) {
	    redisTemplate.boundHashOps(key).put(field, value);
	}
	
	/**
     * 向key对应的map中添加缓存对象
     * @param key   cache对象key
     * @param field map对应的key
     * @param time 过期时间-整个MAP的过期时间
     * @param value     值
     */
	public static  void addMap(String key, String field, Object value, long time) {
	    redisTemplate.boundHashOps(key).put(field, value);
	    redisTemplate.boundHashOps(key).expire(time, TimeUnit.SECONDS);
	}
	
	/**
     * 处理事务时锁定key
     * @param key
     */
	public static  void watch(String key) {
	    redisTemplate.watch(key);
	}
	
	/**
     * 向set中加入对象
     * @param key  对象key
     * @param obj  值
     */
	public static  void addSet(String key, Object... obj) {
	    redisTemplate.boundSetOps(key).add(obj);
	}
	
	 /**
     * 移除set中的某些值
     * @param key  对象key
     * @param obj  值
     */
	public static  long removeSetValue(String key, Object obj) {
	    return redisTemplate.boundSetOps(key).remove(obj);
	}
	
	/**
     * 移除set中的某些值
     * @param key  对象key
     * @param obj  值
     */
	public static  long removeSetValue(String key, Object... obj) {
	    if (obj != null && obj.length > 0) {
	        return redisTemplate.boundSetOps(key).remove(obj);
	    }
	    return 0L;
	}
	
	/**
     * 获取set的对象数
     * @param key  对象key
     */
	public static  long getSetSize(String key) {
	    return redisTemplate.boundSetOps(key).size();
	}
	
	/**
     * 判断set中是否存在这个值
     * @param key  对象key
     */
	public static  Boolean hasSetValue(String key, Object obj) {
	    Boolean boo = null;
	    int t =0;
	    while (true){
	        try {
	            boo = redisTemplate.boundSetOps(key).isMember(obj);
	            break;
	        } catch (Exception e) {
	            t++;
	        }
	        if(t>times){
	            break;
	        }
	    }
	    return boo;
	}
	
	/**
     * 获得整个set
     * @param key  对象key
     */
	public static  Set<Object> getSet(String key) {
	    return redisTemplate.boundSetOps(key).members();
	}
	
	/**
     * 获得set 并集
     * @param key
     * @param otherKey
     * @return
     */
	public static  Set<Object> getSetUnion(String key, String otherKey) {
	    return redisTemplate.boundSetOps(key).union(otherKey);
	}
	
	/**
     * 获得set 并集
     * @param key
     * @param set
     * @return
     */
	public static  Set<Object> getSetUnion(String key, Set<Object> set) {
	    return redisTemplate.boundSetOps(key).union(set);
	}
	
	/**
     * 获得set 交集
     * @param key
     * @param otherKey
     * @return
     */
	public static  Set<Object> getSetIntersect(String key, String otherKey) {
	    return redisTemplate.boundSetOps(key).intersect(otherKey);
	}
	
	/**
     * 获得set 交集
     * @param key
     * @param set
     * @return
     */
	public static  Set<Object> getSetIntersect(String key, Set<Object> set) {
	    return redisTemplate.boundSetOps(key).intersect(set);
	}
	
	
	public static void publish(String channel,Object message){
			Assert.hasText(channel, "a non-empty channel is required");

			final byte[] rawChannel = redisTemplate.getStringSerializer().serialize(channel);
			final byte[] rawMessage;
			if(message instanceof byte[]){
				rawMessage = (byte[]) message;
			}else{
				rawMessage = redisTemplate.getValueSerializer().serialize(message);
			}
			redisTemplate.execute(new RedisCallback<Object>() {

				public Object doInRedis(RedisConnection connection) {
					connection.publish(rawChannel, rawMessage);
					return null;
				}
			}, true);
	}

	public static Long incrementHashValue(String key, String field, long delta) {
		return redisTemplate.boundHashOps(key).increment(field, delta);
	}
	
}
