package com.viontech.mall.report.service.impl;

import com.viontech.keliu.chart.Chart;
import com.viontech.keliu.chart.Table;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.axis.TableHead;
import com.viontech.keliu.chart.factory.AxisFactory;
import com.viontech.keliu.chart.series.SeriesType;
import com.viontech.keliu.i18n.util.LocalMessageUtil;
import com.viontech.keliu.util.DateUtil;
import com.viontech.keliu.util.NumberUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.base.ChartReportBaseService;
import com.viontech.mall.report.enums.KPIType;
import com.viontech.mall.report.enums.OrgType;
import com.viontech.mall.report.service.adapter.FloorReportDataService;
import com.viontech.mall.report.service.adapter.GateReportDataService;
import com.viontech.mall.report.service.adapter.MallReportDataService;
import com.viontech.mall.report.service.adapter.ZoneReportDataService;
import com.viontech.mall.service.adapter.WeatherService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.Map.Entry;

@Service
public class WeatherReportServiceImpl extends ChartReportBaseService {

	@Resource
	private MallReportDataService mallReportDataService;

	@Resource
	private ZoneReportDataService zoneReportDataService;
	
	@Resource
	private FloorReportDataService floorReportDataService;
	
	@Resource
	private GateReportDataService gateReportDataService;
	
	@Resource
	private WeatherService weatherService;
	
	/**天气分析图*/
	public static final String REPORT_WEATHER_ANALYZE ="WeatherAnalyze";
	/**天气综合列表*/
	public static final String REPORT_WEATHER_SYNTHETICAL ="WeatherSynthetical";
	/**每日详情*/
	public static final String REPORT_WEATHER_DETAIL ="WeatherDetail";
	
	/**天气数据*/
	public static final String REPORT_WEATHER_DATA ="WeatherData";
	
	@Override
	public Chart getChart(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart) {
		OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
		Chart chart = null;
		List<Data> datas = new ArrayList<Data>();
		List<Date> dates = DateUtil.getDaysBetweenDates(startDate, endDate);
		
		if (orgType == OrgType.zone) {
			datas = getZoneDatas(orgIds, startDate, endDate, dataMap);
		}else if (orgType == OrgType.gate) {
			datas = getGateDatas(orgIds, startDate, endDate, dataMap);
		}else if (orgType == OrgType.floor) {
			datas = getFloorDatas(orgIds, startDate, endDate, dataMap);
		}else if (orgType == OrgType.mall) {
			datas = getMallDatas(orgIds, startDate, endDate, dataMap);
		}
		switch (reportChart.getKey()) {
			case REPORT_WEATHER_ANALYZE:
				chart = weatherAnalyzeReport(datas, startDate, endDate, dataMap, reportChart);
				break;
			case REPORT_WEATHER_SYNTHETICAL:
				chart = weatherSyntheticalReport(datas, startDate, endDate, dataMap, reportChart);
				break;
			case REPORT_WEATHER_DETAIL:
				chart = weatherDetailReport(datas, startDate, endDate, dataMap, reportChart);
				break;
			default:
				break;
		}
		
		return chart;
	}
	/**天气综合列表*/
	private Chart weatherSyntheticalReport(List<Data> datas, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart){
		Table table = new Table(reportChart.getTitle(),SeriesType.table);
		
		final String weatherStr = LocalMessageUtil.getMessage("weather");//国际化：天气
		final String dayNumStr = LocalMessageUtil.getMessage("dayNum");//国际化：天数
		final String avgAQIStr = LocalMessageUtil.getMessage("avgAQI");//国际化：平均AQI指数
		final String avgTempStr = LocalMessageUtil.getMessage("avgTemp");//国际化：平均温度
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		
		TableHead tableHead = new TableHead();
		tableHead.addData(weatherStr,dayNumStr,avgAQIStr,avgTempStr,kpiType.toString());
		table.setTableHead(tableHead);
		
		putDate2SyntheticalReport(datas, dataMap, table);
		
		return table;
	}
	
	/**天气每日详情列表*/
	public Chart weatherDetailReport(List<Data> datas, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart){
		Table table = new Table(reportChart.getTitle(), SeriesType.table);
		
		final String dateStr = LocalMessageUtil.getMessage("date");//国际化：日期
		final String weekStr = LocalMessageUtil.getMessage("week");//国际化：星期
		final String weatherStr = LocalMessageUtil.getMessage("weather");//国际化：天气
		final String AQIStr = LocalMessageUtil.getMessage("AQI");//国际化：AQI指数
		final String TempStr = LocalMessageUtil.getMessage("Temp");//国际化：温度
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		
		TableHead tableHead = new TableHead();
		tableHead.addData(dateStr,weekStr,weatherStr,AQIStr,TempStr,kpiType.toString());
		table.setTableHead(tableHead);
		
		putData2DetailChart(datas, dataMap, table);
		
		return table;
	}
	
	/**天气分析图*/
	private Chart weatherAnalyzeReport(List<Data> datas, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart){
		Chart chart = new Chart(reportChart.getTitle(), SeriesType.scatter);
		
		Axis xAxis = AxisFactory.createDayOFMonthAxis(startDate, endDate);
		chart.setXAxis(xAxis);
		
		putData2AnalyzeChart(datas, dataMap,chart);
		return chart;
	}
	
	/**
	 * 把数据放入详情报表中
	 * @param dates
	 * @param datas
	 * @param dataMap
	 */
	private void putData2DetailChart(List<Data> datas, Map<String, Object> dataMap,Table table){
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		List<String> fields = table.getTableHead().getData();
		List<List<String>> fieldDatas = new ArrayList<List<String>>();
		for (Data data : datas) {
			List<String> fieldData = new ArrayList<String>();
			Date date = data.getDate();
			String dateStr = DateUtil.format(DateUtil.FORMAT_MM_DD,date);
			fieldData.add(dateStr);
			String weekStr = DateUtil.format(DateUtil.FORMAT_DAY_OF_WEEK_SORT_CN, date);
			fieldData.add(weekStr);
			Weather weather = data.getWeather();
			if (weather == null) {
				continue ;
			}
			String weatherType = weather.getType();
			fieldData.add(weatherType);
			String AQI = ""+weather.getAqi();
			fieldData.add(AQI);
			String temp = ""+weather.getLtemp()+"~"+weather.getHtemp();
			fieldData.add(temp);
			Object result = null;
			if (kpiType == KPIType.TRAFFIC || kpiType == KPIType.SALES ) {
				result = data.getValue1();
			}else if (kpiType == KPIType.ENTERINGRATE) {
				result = NumberUtil.percentage(Double.parseDouble(data.getValue1().toString()),
									Double.parseDouble(data.getValue2().toString()), 2);
			}else if (kpiType == KPIType.PREPRICE  || kpiType == KPIType.PERAREAVALUE) {
				result = NumberUtil.divide(Double.parseDouble(data.getValue1().toString()),Double.parseDouble(data.getValue2().toString()), 2);
			}else if (kpiType == KPIType.ORDER) {
				result = data.getValue2();
			}else if (kpiType == KPIType.HANDBAGRATE) {
				result = NumberUtil.percentage(Double.parseDouble(data.getValue2().toString()),
						Double.parseDouble(data.getValue1().toString()), 2);
			}else if (kpiType == KPIType.DURATIONTIME) {
				result = NumberUtil.divide(Double.parseDouble(data.getValue2().toString()),Double.parseDouble(data.getValue1().toString()), 2);

			}
			fieldData.add(result.toString());
			fieldDatas.add(fieldData);
		}
		for (List<String> fieldData : fieldDatas) {
			for (int i = 0; i < fields.size(); i++) {
				table.getRow(fieldData.get(0)).putValueByHeadColumn(fields.get(i), fieldData.get(i));
			}
		}
	}
	
	/**
	 * 把数据放入分析报表中
	 * @param dates
	 * @param datas
	 * @param dataMap
	 */
	private void putData2AnalyzeChart(List<Data> datas, Map<String, Object> dataMap,Chart chart){
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		for (Data data : datas) {
			Weather weather = data.getWeather();
			if (weather == null) {
				continue;
			}
			String weatherType = weather.getType();
			Date date = data.getDate();
			Object result = null;
			if (kpiType == KPIType.TRAFFIC || kpiType == KPIType.SALES ) {
				result = data.getValue1();
			}else if (kpiType == KPIType.ENTERINGRATE) {
				result = NumberUtil.percentage(Double.parseDouble(data.getValue1().toString()),
									Double.parseDouble(data.getValue2().toString()), 2);
			}else if (kpiType == KPIType.PREPRICE || kpiType == KPIType.PERAREAVALUE) {
				result = NumberUtil.divide(Double.parseDouble(data.getValue1().toString()),Double.parseDouble(data.getValue2().toString()), 2);
			}else if (kpiType == KPIType.ORDER) {
				result = data.getValue2();
			}else if (kpiType == KPIType.HANDBAGRATE) {
				result = NumberUtil.percentage(Double.parseDouble(data.getValue2().toString()),
						Double.parseDouble(data.getValue1().toString()), 2);
			}else if (kpiType == KPIType.DURATIONTIME) {
				result = NumberUtil.divide(Double.parseDouble(data.getValue2().toString()),Double.parseDouble(data.getValue1().toString()), 2);
			}
			data.setValue1(result);
			chart.getSeries(weatherType).putValueByCoordinate(date, data);
		}
	}
	
	/**
	 * 把数据放入天气综合列表
	 * @param datas   原始数据
	 * @param dataMap  内存缓存数据
	 * @param chart
	 */
	@SuppressWarnings("rawtypes")
	private void putDate2SyntheticalReport(List<Data> datas, Map<String, Object> dataMap,Table table){
		
		Map<String, List<Integer>> ltempMap = new HashMap<String, List<Integer>>();
		Map<String, List<Integer>> htempMap = new HashMap<String, List<Integer>>();
		Map<String, List<Integer>> AQIMap = new HashMap<String, List<Integer>>();
		Map<String, Double> value1Map = new HashMap<String, Double>();
		Map<String, Double> value2Map = new HashMap<String, Double>();
		
		for (Data data : datas) {
			Weather weather = data.getWeather();
			if (weather == null) {
				continue;
			}
			int ltemp = weather.getLtemp();
			int htemp = weather.getHtemp();
			String weatherType = weather.getType();
			int AQI = weather.getAqi();
		    double value1 = NumberUtil.parseDouble(data.getValue1());
		    double value2 = NumberUtil.parseDouble(data.getValue2());
			
			List<Integer> ltempList = new ArrayList<Integer>();
			List<Integer> htempList = new ArrayList<Integer>();
			List<Integer> AQIList = new ArrayList<Integer>();
			
			if (ltempMap.get(weatherType) != null) {
				ltempList = ltempMap.get(weatherType);
			}
			ltempList.add(ltemp);
			ltempMap.put(weatherType, ltempList);
			if (htempMap.get(weatherType) != null) {
				htempList = htempMap.get(weatherType);
			}
			htempList.add(htemp);
			htempMap.put(weatherType, htempList);
			if (AQIMap.get(weatherType) != null) {
				AQIList = AQIMap.get(weatherType);
			}
			AQIList.add(AQI);
			AQIMap.put(weatherType, AQIList);
			
			if (value1Map.get(weatherType) != null) {
				value1 += value1Map.get(weatherType);
			}
			value1Map.put(weatherType, value1);
			if (value2Map.get(weatherType) != null) {
				value2 += value2Map.get(weatherType);
			}
			value2Map.put(weatherType, value2);
		}
		List<String> fields = table.getTableHead().getData();
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		for (Entry<String, List<Integer>> entity : ltempMap.entrySet()) {
			String weatherType = entity.getKey();
			table.getRow(weatherType).putValueByHeadColumn(fields.get(0), weatherType); //天气
			table.getRow(weatherType).putValueByHeadColumn(fields.get(1), ltempMap.get(weatherType).size()); //天数
			int dayNum = entity.getValue().size();
			int aqiNum = 0;
			List<Integer> aqiList = AQIMap.get(weatherType);
			for (int i = 0; i < aqiList.size(); i++) {
				aqiNum +=aqiList.get(i);
			}
			double perAQI = NumberUtil.divide(aqiNum, aqiList.size(), 2);
			table.getRow(weatherType).putValueByHeadColumn(fields.get(2), perAQI); //平均AQI指数
			
			int lthmpNum = 0;
			List<Integer> ltempList = ltempMap.get(weatherType);
			for (int i = 0; i < ltempList.size(); i++) {
				lthmpNum += ltempList.get(i);
			}
			double perltemp = NumberUtil.divide(lthmpNum, ltempList.size(), 1);
			
			int hthmpNum = 0;
			List<Integer> htempList = htempMap.get(weatherType);
			for (int i = 0; i < htempList.size(); i++) {
				hthmpNum += htempList.get(i);
			}
			double perhtemp = NumberUtil.divide(hthmpNum, htempList.size(), 1);
			table.getRow(weatherType).putValueByHeadColumn(fields.get(3), perltemp +"~"+ perhtemp ); //平均温度范围
			
			Object result = null;
			if (kpiType == KPIType.TRAFFIC || kpiType == KPIType.SALES) {
				result = value1Map.get(weatherType);
			}else if (kpiType == KPIType.ENTERINGRATE) {
				result = NumberUtil.percentage(value1Map.get(weatherType), value2Map.get(weatherType), 2);
			}else if (kpiType == KPIType.PREPRICE) {
				result = NumberUtil.divide(value1Map.get(weatherType), value2Map.get(weatherType), 2);
			}else if (kpiType == KPIType.PERAREAVALUE) {
				Double area = NumberUtil.divide(value2Map.get(weatherType), value2Map.size(), 2);
				result = NumberUtil.divide(value1Map.get(weatherType),area, 2);
			}else if (kpiType == KPIType.ORDER) {
				result = value2Map.get(weatherType);
			}else if (kpiType == KPIType.HANDBAGRATE ) {
				result = NumberUtil.percentage(value2Map.get(weatherType), value1Map.get(weatherType), 2);
			}else if (kpiType == KPIType.DURATIONTIME) {
				result = NumberUtil.divide(value2Map.get(weatherType), value1Map.get(weatherType), 2);
			}
			table.getRow(weatherType).putValueByHeadColumn(fields.get(4), result); 
		}
		
		
	}
	
	public List<Data> getMallDatas(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		Map<Date, Weather> weatherMaps = getWeatherData(orgIds, startDate, endDate, dataMap);
		List<MallDayCountData> trafficDatas = new ArrayList<MallDayCountData>();
		List<Sale> saleDatas = new ArrayList<Sale>();
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		if (kpiType == KPIType.TRAFFIC ) {
			trafficDatas = mallReportDataService.getOrQueryMallBetweenDateDayTraffic(orgIds, startDate, endDate, dataMap);
			for (MallDayCountData dayCountData : trafficDatas) {
				Date date = dayCountData.getCountdate();
				int innum = dayCountData.getInnum();
				Weather weather = weatherMaps.get(date);
				Data data = new Data(date, weather, innum, null);
				datas.add(data);
			}
		}else if (kpiType == KPIType.SALES || kpiType == KPIType.PREPRICE || kpiType == KPIType.PERAREAVALUE ) {
			saleDatas = mallReportDataService.getOrQueryMallBetweenDateDaySale(orgIds, startDate, endDate, dataMap);
			for (Sale sale : saleDatas) {
				Date date = sale.getSaledate();
				Double saleMoney = sale.getMoney();
				int saleCount = sale.getSalecount();
				Weather weather = weatherMaps.get(date);
				Data data = null;
				if (kpiType == KPIType.PERAREAVALUE) {
					Long mallId = sale.getMallId();
					Mall mall = mallReportDataService.getMallByMallId(mallId, dataMap);
					Float area = null;
					if (mall != null && mall.getArea()!=null && mall.getArea() != 0) {
						area = mall.getArea();
					}
					data = new Data(date, weather, saleMoney, area);
				}else {
					data = new Data(date, weather, saleMoney, saleCount);
				}
				datas.add(data);
			}
		}
		return datas;
	}
	
	public List<Data> getFloorDatas(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		Map<Date, Weather> weatherMaps = getWeatherData(orgIds, startDate, endDate, dataMap);
		List<FloorDayCountData> trafficDatas = new ArrayList<FloorDayCountData>();
		List<Sale> saleDatas = new ArrayList<Sale>();
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		if (kpiType == KPIType.TRAFFIC ) {
			trafficDatas = floorReportDataService.getOrQueryFloorBetweenDateDayTraffic(orgIds, startDate, endDate, dataMap);
			for (FloorDayCountData dayCountData : trafficDatas) {
				Date date = dayCountData.getCountdate();
				int innum = dayCountData.getInnum();
				Weather weather = weatherMaps.get(date);
				Data data = new Data(date, weather, innum, null);
				datas.add(data);
			}
		}else if (kpiType == KPIType.SALES) {
			saleDatas = mallReportDataService.getOrQueryMallBetweenDateDaySale(orgIds, startDate, endDate, dataMap);
			for (Sale sale : saleDatas) {
				Date date = sale.getSaledate();
				Double saleMoney = sale.getMoney();
				int saleCount = sale.getSalecount();
				Weather weather = weatherMaps.get(date);
				Data data = new Data(date, weather, saleMoney, saleCount);
				datas.add(data);
			}
		}
		return datas;
	}
	
 	public List<Data> getZoneDatas(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		Map<Date, Weather> weatherMaps = getWeatherData(orgIds, startDate, endDate, dataMap);
		List<ZoneDayCountData> trafficDatas = new ArrayList<ZoneDayCountData>();
		List<ZoneHourCountData> trafficHourDatas = new ArrayList<ZoneHourCountData>();
		List<Sale> saleDatas = new ArrayList<Sale>();
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		if (kpiType == KPIType.TRAFFIC || kpiType == KPIType.ENTERINGRATE || kpiType == KPIType.HANDBAGRATE || kpiType == KPIType.DURATIONTIME) {
			trafficDatas = zoneReportDataService.getOrQueryZoneBetweenDateDayTraffic(orgIds, startDate, endDate, dataMap);
			for (ZoneDayCountData dayCountData : trafficDatas) {
				Date date = dayCountData.getCountdate();
				int passNum = dayCountData.getOutsideInnum()+dayCountData.getOutsideOutnum();
				int innum = dayCountData.getInnum();
				Weather weather = weatherMaps.get(date);
				Data data = new Data(date, weather, innum, passNum);
				datas.add(data);
			}
		}else if (kpiType == KPIType.SALES || kpiType == KPIType.PREPRICE || kpiType == KPIType.PERAREAVALUE || kpiType == KPIType.ORDER || kpiType == KPIType.HANDBAGRATE) {
			saleDatas = zoneReportDataService.getOrQueryZoneBetweenDateSale(orgIds, startDate, endDate, dataMap);
			for (Sale sale : saleDatas) {
				Date date = sale.getSaledate();
				Double saleMoney = sale.getMoney();
				int saleCount = sale.getSalecount();
				Weather weather = weatherMaps.get(date);
				Data data = null;
				if (kpiType == KPIType.PERAREAVALUE) {
					Long zoneId = sale.getZoneId();
					Zone zone = zoneReportDataService.getOrQueryZoneById(zoneId, dataMap);
					Float area = null;
					if (zone != null && zone.getArea()!=null && zone.getArea() != 0) {
						area = zone.getArea();
					}
					data = new Data(date, weather, saleMoney, area);
				}else if (kpiType == KPIType.HANDBAGRATE) {
					for (Data vdata : datas) {
						if (DateUtil.isSameDay(date, vdata.getDate())) {
							vdata.setValue2(saleCount);
						}
					}
				}else {
					data = new Data(date, weather, saleMoney, saleCount);
				}
				datas.add(data);
			}
		}else if (kpiType == KPIType.DURATIONTIME) {
			trafficHourDatas = zoneReportDataService.getOrQueryZoneBetweenDateHourTraffic(orgIds, startDate, endDate, dataMap);
			Map<Date, Integer> hourInnumAddMap = new HashMap<Date, Integer>();
			for (ZoneHourCountData hourCountData : trafficHourDatas) {
				Date date = hourCountData.getCountdate();
				int innum = hourCountData.getInnum();
				if (hourInnumAddMap.get(date) == null) {
					hourInnumAddMap.put(date, 0);
				}
				innum += hourInnumAddMap.get(date);
				hourInnumAddMap.put(date, innum);
			}
			for (Data data : datas) {
				Date date = data.getDate();
				data.setValue2(hourInnumAddMap.get(date));
			}
		}
		return datas;
	}
 	
	public List<Data> getGateDatas(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		Map<Date, Weather> weatherMaps = getWeatherData(orgIds, startDate, endDate, dataMap);
		List<GateDayCountData> trafficDatas = new ArrayList<GateDayCountData>();
		KPIType kpiType = (KPIType) dataMap.get("KPITYPE");
		if (kpiType == KPIType.TRAFFIC ) {
			trafficDatas = gateReportDataService.getOrQueryGateBetweenDateDayTraffic(orgIds, startDate, endDate, dataMap);
			for (GateDayCountData dayCountData : trafficDatas) {
				Date date = dayCountData.getCountdate();
				int innum = dayCountData.getInnum();
				Weather weather = weatherMaps.get(date);
				Data data = new Data(date, weather, innum, null);
				datas.add(data);
			}
		}
		return datas;
	}
	
	/**
	 * 获取机构所在城市的天气数据
	 * @param orgIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private Map<Date,Weather> getWeatherData(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		Map<Date,Weather>  weatherMaps = (Map<Date, Weather>) dataMap.get(REPORT_WEATHER_DATA);
		if (weatherMaps == null) {
			weatherMaps = new HashMap<Date, Weather>();
			if (orgIds == null || orgIds.length <= 0) {
				return weatherMaps;
			}
			OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
			Long cityId = null;
			Long mallId = null;
			if (orgType == OrgType.zone ) {
				Long zoneId = orgIds[0];
				Zone zone = zoneReportDataService.getOrQueryZoneById(zoneId, dataMap);
			    mallId = zone.getMallId();
			}else if (orgType == OrgType.floor) {
				Long floorId = orgIds[0];
				Floor floor = floorReportDataService.getFloorInfoById(floorId, dataMap);
				mallId = floor.getMallId();
			}else if (orgType == OrgType.gate) {
				Long gateId = orgIds[0];
				Gate gate = gateReportDataService.getGateInfoById(gateId, dataMap);
				mallId = gate.getMallId();
			}else if (orgType == OrgType.mall) {
				mallId = orgIds[0];
			}
			Mall mall = mallReportDataService.getMallByMallId(mallId, dataMap);
			if(mall == null || mall.getCityId() == null)
				 return weatherMaps;
			cityId =  mall.getCityId();
			WeatherExample example = new WeatherExample();
			example.createCriteria().andCityIdEqualTo(cityId).andDataDateBetween(startDate, endDate);
			List<Weather> weathers = weatherService.selectByExample(example);
			if (weathers != null && weathers.size() > 0) {
				for (Weather weather : weathers) {
					Date date = weather.getDataDate();
					weatherMaps.put(date, weather);
				}
			}
			dataMap.put(REPORT_WEATHER_DATA, weatherMaps);
		}
		return weatherMaps;
	}
	
	@Override
	public Map<String, Object> getHead(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap) {
		return null;
	}
	
	class Data<T,O> {
		private Date date;
		private Weather weather;
		private T value1;
		private O value2;
		
		public Data(Date date, Weather weather, T value1, O value2) {
			this.date = date;
			this.weather = weather;
			this.value1 = value1;
			this.value2 = value2;
		}
		
		public Date getDate() {
			return date;
		}
		public void setDate(Date date) {
			this.date = date;
		}
		public Weather getWeather() {
			return weather;
		}
		public void setWeather(Weather weather) {
			this.weather = weather;
		}
		public T getValue1() {
			return value1;
		}
		public void setValue1(T value1) {
			this.value1 = value1;
		}
		public O getValue2() {
			return value2;
		}
		public void setValue2(O value2) {
			this.value2 = value2;
		}
	}
}
