package com.viontech.mall.report.service.impl;

import com.viontech.keliu.chart.Chart;
import com.viontech.keliu.chart.Table;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.axis.TableHead;
import com.viontech.keliu.chart.factory.AxisFactory;
import com.viontech.keliu.chart.series.SeriesType;
import com.viontech.keliu.i18n.util.LocalMessageUtil;
import com.viontech.keliu.util.NumberUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.base.ChartReportBaseService;
import com.viontech.mall.report.enums.OrgType;
import com.viontech.mall.report.util.AgeProcessUtil;
import com.viontech.mall.service.adapter.ZoneDayFaceRecognitionStaService;
import com.viontech.mall.service.adapter.ZoneService;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

@Service
public class TrafficAnalyzeReportServiceImpl extends ChartReportBaseService {

    @Value("${Age.stage}")
    private String ageStage;

    private final static String FELID_ZONES="zones";
    private final static String FELID_FACE_DATA="faceDatas";

    /**
     * 区域人脸客流统计详情
     */
    private final static  String  REPORT_FACE_TRAFFIC_DETAIL = "faceZoneTrafficDetail";
    /**
     * 区域人脸性别比例统计
     */
    private final static  String  REPORT_FACE_GENGER_RATIO = "faceZoneGenderRatio";
    /**
     * 区域人脸年龄比例统计
     */
    private final static  String  REPORT_FACE_AGE_RATIO = "faceZoneAgeRatio";

    @Resource
    private ZoneDayFaceRecognitionStaService zoneDayFaceRecognitionStaService;

    @Resource
    private ZoneService zoneService;
    @Override
    public Map<String, Object> getHead(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap) {
        return null;
    }

    @Override
    public Chart getChart(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart) {

        List<ZoneDayFaceRecognitionSta> faceDatas = getZoneDayFaceDatas(orgIds, startDate, endDate, dataMap);

        Chart chart = null;
        switch (reportChart.getKey()) {
            case REPORT_FACE_TRAFFIC_DETAIL:
                chart = faceTrafficDatailReport(orgIds, faceDatas, dataMap, reportChart);
                break;
            case REPORT_FACE_GENGER_RATIO:
                chart = faceGenderRatioReport(orgIds, faceDatas, dataMap, reportChart);
                break;
            case REPORT_FACE_AGE_RATIO:
                chart = faceAgeRatioReport(orgIds, faceDatas, dataMap, reportChart);
                break;
            default:
                break;
        }
        return chart;
    }

    /**
     * 区域人脸客流统计详情
     * @param orgIds
     * @param faceDatas
     * @param dataMap
     * @param reportChart
     * @return
     */
    private Chart faceTrafficDatailReport(Long[] orgIds,List<ZoneDayFaceRecognitionSta> faceDatas,Map<String, Object> dataMap, ReportChart reportChart){
        final String zoneName = LocalMessageUtil.getMessage("Face.zoneName");
        final String allTime = LocalMessageUtil.getMessage("Face.allTime");
        final String allCount = LocalMessageUtil.getMessage("Face.allCount");
        final String staffTime = LocalMessageUtil.getMessage("Face.staffTime");
        final String staffCount = LocalMessageUtil.getMessage("Face.staffCount");
        final String costomerTime = LocalMessageUtil.getMessage("Face.customerTime");
        final String customerCount = LocalMessageUtil.getMessage("Face.customerCount");


        Table table = new Table(reportChart.getTitle());
        TableHead tableHead = new TableHead();
        tableHead.addData(zoneName);
        tableHead.addData(allTime);
        tableHead.addData(allCount);
        tableHead.addData(staffTime);
        tableHead.addData(staffCount);
        tableHead.addData(costomerTime);
        tableHead.addData(customerCount);
        table.setTableHead(tableHead);

        //区域放入表格第一列，（预防区域无数据时，区域名称不显示的问题）
        Map<Long, Zone> zones = getZones(orgIds, dataMap);
        for (Zone zone : zones.values()) {
            table.getRow(zone.getName()).putValueByHeadColumn(zoneName,zone.getName());
        }

        //人脸数据
        for (ZoneDayFaceRecognitionSta face : faceDatas) {
            Long gateId = face.getZoneId();
            Zone zone = zones.get(gateId);
            table.getRow(zone.getName()).putValueByHeadColumn(zoneName, zone.getName());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(allTime,face.getPersonMantime());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(allCount,face.getPersonCount());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(staffTime,face.getStaffMantime());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(staffCount,face.getStaffCount());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(costomerTime, face.getCustomMantime());
            table.getRow(zone.getName()).adjustOrPutValueByHeadColumn(customerCount,face.getCustomCount());
        }
        return table;
    }

    /**
     * 人脸性别统计
     * @param orgIds
     * @param faceDatas
     * @param dataMap
     * @param reportChart
     * @return
     */
    private Chart faceGenderRatioReport(Long[] orgIds,List<ZoneDayFaceRecognitionSta> faceDatas, Map<String, Object> dataMap, ReportChart reportChart){
        final String male = LocalMessageUtil.getMessage("ParamName.male");
        final String female = LocalMessageUtil.getMessage("ParamName.female");

        Chart chart = new Chart(reportChart.getTitle(), SeriesType.bar);
        Axis<String> xAxis = AxisFactory.createStringAxis();
        Map<Long, Zone> zones = getZones(orgIds, dataMap);
        for (Zone zone : zones.values()) {
            xAxis.addData(zone.getName());
        }
        chart.setXAxis(xAxis);

        for (ZoneDayFaceRecognitionSta face : faceDatas) {
            Long zoneId = face.getZoneId();
            Zone zone = zones.get(zoneId);
            chart.getSeries(male).adjustOrPutValueByCoordinate(zone.getName(), face.getMaleCount());
            chart.getSeries(female).adjustOrPutValueByCoordinate(zone.getName(), face.getFemaleCount());
        }

        return chart;
    }

    /**
     * 人脸年龄统计详情
     * @param orgIds
     * @param faceDatas
     * @param dataMap
     * @param reportChart
     * @return
     */
    private Chart faceAgeRatioReport(Long[] orgIds,List<ZoneDayFaceRecognitionSta> faceDatas, Map<String, Object> dataMap, ReportChart reportChart){
        Chart chart = new Chart(reportChart.getTitle(), SeriesType.bar){
            @Override
            public Object calcValue(String name, int index, List data_raw) {
                if(data_raw.get(index) == null)
                    return 0;
                return super.calcValue(name, index, data_raw);
            }
        };

        Axis<String> xAxis = AxisFactory.createStringAxis();
        Map<Long, Zone> zones = getZones(orgIds, dataMap);
        for (Zone zone : zones.values()) {
            xAxis.addData(zone.getName());
        }
        chart.setXAxis(xAxis);

        String[] ages = ageStage.split(",");
        List<Integer> ageStages = new ArrayList<>();
        for (String age : ages) {
            Integer ageInt = Integer.parseInt(age);
            ageStages.add(ageInt);
        }
        String[] ageThresholdName = AgeProcessUtil.calAgeThresholdName(ageStages);
        for (String ageStr : ageThresholdName) {
            chart.createSeries(ageStr);
        }

        for (ZoneDayFaceRecognitionSta face : faceDatas) {
            Long zoneId = face.getZoneId();
            Zone zone = zones.get(zoneId);

            String maleStage = face.getMaleStage();
            String femaleStage = face.getFemaleStage();
            if( maleStage==null ||  maleStage.isEmpty() ||  femaleStage==null || femaleStage.isEmpty()){
                continue;
            }
            String[] maleAgestatic = maleStage.split(",", -2);
            String[] femaleAgestatic = femaleStage.split(",", -2);
            if (maleAgestatic.length != femaleAgestatic.length) {
                continue;
            }
            for (int rangeNum = 0; rangeNum < maleAgestatic.length; rangeNum++) {
                int maleNum = Integer.parseInt("" + maleAgestatic[rangeNum].trim());
                int femaleNum = Integer.parseInt("" + femaleAgestatic[rangeNum].trim());
                String ageRange = ageThresholdName[AgeProcessUtil.getIndexByAge(rangeNum,ageStages)];
                chart.getSeries(ageRange).adjustOrPutValueByCoordinate(zone.getName(), NumberUtil.valueAdd(femaleNum,maleNum));
                //chart.getSeries(reportChart.getTitle()).adjustOrPutValueByCoordinate(ageRange, NumberUtil.valueAdd(femaleNum,maleNum));
            }

        }

        return chart;
    }

    /**
     * 获得区域数据
     * @param orgIds
     * @param dataMap
     * @return
     */
    private Map<Long,Zone> getZones(Long[] orgIds,Map<String, Object> dataMap){
        Map<Long,Zone> zoneMap =  (Map<Long, Zone>) dataMap.get(FELID_ZONES);
        if(zoneMap == null){
            zoneMap = new HashMap<Long, Zone>();
            OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
            ZoneExample zoneExample = new ZoneExample();
            if(orgType == OrgType.mall){
                zoneExample.createCriteria().andMallIdIn(Arrays.asList(orgIds));

            }else if(orgType == OrgType.zone){
                zoneExample.createCriteria().andIdIn(Arrays.asList(orgIds));
            }
            List<Zone> zones = zoneService.selectByExample(zoneExample);
            if(zones == null){
                zones = new ArrayList<Zone>();
            }
            for (Zone zone : zones) {
                zoneMap.put(zone.getId(), zone);
            }
            dataMap.put(FELID_ZONES, zoneMap);
        }
        return zoneMap;
    }


    private List<ZoneDayFaceRecognitionSta> getZoneDayFaceDatas(Long[] orgIds,Date startDate,Date endDate,Map<String, Object> dataMap){
        List<ZoneDayFaceRecognitionSta> faceDatas = (List<ZoneDayFaceRecognitionSta>) dataMap.get(FELID_FACE_DATA);
        if (faceDatas == null){
            faceDatas = new ArrayList<>();
            ZoneDayFaceRecognitionStaExample zoneDayFaceRecognitionStaExample = new ZoneDayFaceRecognitionStaExample();
            zoneDayFaceRecognitionStaExample.createCriteria().andZoneIdIn(Arrays.asList(orgIds)).andCountdateGreaterThanOrEqualTo(startDate)
                    .andCountdateLessThanOrEqualTo(endDate);
            faceDatas = zoneDayFaceRecognitionStaService.selectByExample(zoneDayFaceRecognitionStaExample);
            dataMap.put(FELID_FACE_DATA,faceDatas);
        }
        return  faceDatas;
    }
}
