package com.viontech.mall.report.service.impl;

import com.viontech.keliu.chart.Chart;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.factory.AxisFactory;
import com.viontech.keliu.chart.series.SeriesType;
import com.viontech.keliu.util.DateUtil;
import com.viontech.keliu.util.NumberUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.base.BaseDataServiceImpl.DateCriteria;
import com.viontech.mall.report.base.ChartReportBaseService;
import com.viontech.mall.report.enums.OrgType;
import com.viontech.mall.report.service.adapter.FloorReportDataService;
import com.viontech.mall.report.service.adapter.GateReportDataService;
import com.viontech.mall.report.service.adapter.MallReportDataService;
import com.viontech.mall.report.service.adapter.ZoneReportDataService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

@Service
public class ThermodynamicReportServiceImpl extends ChartReportBaseService {
	
	@Resource
	private GateReportDataService gateReportDataService;
	
	@Resource
	private MallReportDataService mallReportDataService;

	@Resource
	private ZoneReportDataService zoneReportDataService;
	
	@Resource
	private FloorReportDataService floorReportDataService;
	
	/**时间热力图*/
	public static final String REPORT_TIME_THERMODYNAMIC ="TimeThermodynamic";

	@Override
	public Chart getChart(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart) {
		OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
		List<Data> datas = new ArrayList<Data>();
		Chart chart = null;
		switch (orgType) {
		case gate:
			datas = getThermodynamicGateData(orgIds, startDate, endDate, dataMap);
			break;
		case zone:
			datas = getThermodynamicZoneData(orgIds, startDate, endDate, dataMap);
			break;
		case floor:
			datas = getThermodynamicFloorData(orgIds, startDate, endDate, dataMap);
			break;
		case mall:
			datas = getThermodynamicMallData(orgIds, startDate, endDate, dataMap);
			break;
		default:
			break;
		}
		if (REPORT_TIME_THERMODYNAMIC.equals(reportChart.getKey())) {
			chart = timeThermodynamicChart(startDate, endDate, datas, dataMap, reportChart);
		}
		return chart;
	}
	
	/**
	 * 获取时间热力图的监控点数据
	 * @param orgIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private List<Data> getThermodynamicGateData(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		List<GateHourCountData> hourCountDatas = gateReportDataService.getOrQueryGateHourData(orgIds, startDate, endDate, dataMap);
		if (hourCountDatas == null || hourCountDatas.size() <= 0) {
			return datas ;
		}
		Long mallId = null;
		for (GateHourCountData hourCountData : hourCountDatas) {
			mallId = hourCountData.getMallId();
			if (mallId == null) {
				continue;
			}
			dataMap.put("MALLID", mallId);
			Data<Integer> data = new Data<Integer>(hourCountData.getCounttime(), hourCountData.getCountdate(), hourCountData.getInnum());
			datas.add(data);
		}
		return datas;
	}
	
	/**
	 * 获取时间热力图的店铺数据
	 * @param orgIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private List<Data> getThermodynamicZoneData(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		List<ZoneHourCountData> hourCountDatas = zoneReportDataService.getOrQueryZoneBetweenDateHourTraffic(orgIds, startDate, endDate, dataMap);
		if (hourCountDatas == null || hourCountDatas.size() <= 0) {
			return datas ;
		}
		Long mallId = null;
		for (ZoneHourCountData hourCountData : hourCountDatas) {
			mallId = hourCountData.getMallId();
			if (mallId == null) {
				continue;
			}
			dataMap.put("MALLID", mallId);
			Data<Integer> data = new Data<Integer>(hourCountData.getCounttime(), hourCountData.getCountdate(), hourCountData.getInnum());
			datas.add(data);
		}
		return datas;
	}
	
	/**
	 * 获取时间热力图的楼层数据
	 * @param orgIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private List<Data> getThermodynamicFloorData(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		List<FloorHourCountData> hourCountDatas = floorReportDataService.getOrQueryFloorBetweenDateHourTraffic(orgIds, startDate, endDate, dataMap);
		if (hourCountDatas == null || hourCountDatas.size() <= 0) {
			return datas ;
		}
		Long mallId = null;
		for (FloorHourCountData hourCountData : hourCountDatas) {
			mallId = hourCountData.getMallId();
			if (mallId == null) {
				continue;
			}
			dataMap.put("MALLID", mallId);
			Data<Integer> data = new Data<Integer>(hourCountData.getCounttime(), hourCountData.getCountdate(), hourCountData.getInnum());
			datas.add(data);
		}
		return datas;
	}
	
	/**
	 * 获取时间热力图的商场数据
	 * @param orgIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private List<Data> getThermodynamicMallData(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap){
		List<Data> datas = new ArrayList<Data>();
		List<MallHourCountData> hourCountDatas = mallReportDataService.getOrQueryMallBetweenDateHourTraffic(orgIds, startDate, endDate, dataMap);
		if (hourCountDatas == null || hourCountDatas.size() <= 0) {
			return datas ;
		}
		dataMap.put("MALLID", orgIds[0]);
		for (MallHourCountData hourCountData : hourCountDatas) {
			Data<Integer> data = new Data<Integer>(hourCountData.getCounttime(), hourCountData.getCountdate(), hourCountData.getInnum());
			datas.add(data);
		}
		return datas;
	}
	
	/**
	 *生成时间热力图报表格式
	 * @param startDate
	 * @param endDate
	 * @param datas
	 * @param reportChart
	 * @return
	 */
	private Chart timeThermodynamicChart(Date startDate,Date endDate, List<Data> datas, Map<String, Object> dataMap ,ReportChart reportChart){
		Chart chart = new Chart(reportChart.getTitle(), SeriesType.table);
		
		Long mallId = (Long) dataMap.get("MALLID");
		if (mallId != null) {
			DateCriteria mallOpentime = mallReportDataService.getMallOpentimesByOrgId(mallId, OrgType.mall, dataMap);
			Axis<Date> xAxis = AxisFactory.creatSdfDateAxix("HH:00",Calendar.HOUR);
			xAxis.setMin(mallOpentime.getStartDate());
			xAxis.setMax(mallOpentime.getEndDate());
			xAxis.lockMinMax();
			chart.setXAxis(xAxis);
		}
		
		putValue2Chart(chart,startDate,endDate,datas);
		return chart;
		
	}
	
    @SuppressWarnings("rawtypes")
	private void putValue2Chart(Chart chart,Date startDate,Date endDate,List<Data> datas){
    	List<Date> dates = DateUtil.getDaysBetweenDates(startDate, endDate);
    	Map<String, Integer> weekCountMap = new HashMap<String, Integer>();
    	for (Date date : dates) {
			String weekStr = DateUtil.format(DateUtil.FORMAT_DAY_OF_WEEK_SORT_CN, date);
			if (weekCountMap.get(weekStr) == null) {
				weekCountMap.put(weekStr, 0);
			}
			int countNum = weekCountMap.get(weekStr);
			weekCountMap.put(weekStr, countNum+1);
		}
    	
		for (Data data : datas) {
			Date time = data.getTime();
			Date date = data.getDate();
			String weekStr = DateUtil.format(DateUtil.FORMAT_DAY_OF_WEEK_SORT_CN, date);
			int countNum = 0;
			if(weekCountMap.get(weekStr) != null){
				countNum = weekCountMap.get(weekStr);
			}
			chart.getSeries(weekStr).adjustOrPutValueByCoordinate(time,NumberUtil.divide(NumberUtil.parseInteger(data.getValue()), countNum, 2));
		}
    }

	
	class Data<T>{
		private Date time;
		private Date date;
		private T value;
		
		public Data(Date time, Date date, T value) {
			super();
			this.time = time;
			this.date = date;
			this.value = value;
		}
		
		public Date getTime() {
			return time;
		}
		public void setTime(Date time) {
			this.time = time;
		}
		public Date getDate() {
			return date;
		}
		public void setDate(Date date) {
			this.date = date;
		}
		public T getValue() {
			return value;
		}
		public void setValue(T value) {
			this.value = value;
		}
	}

	@Override
	public Map<String, Object> getHead(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap) {
		return null;
	}
}
