package com.viontech.mall.report.service.impl;

import com.viontech.keliu.i18n.util.LocalMessageUtil;
import com.viontech.keliu.util.DateUtil;
import com.viontech.keliu.util.JsonMessageUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.enums.OrgType;
import com.viontech.mall.report.service.adapter.GateReportDataService;
import com.viontech.mall.report.service.adapter.MovingLineReportService;
import com.viontech.mall.service.adapter.FaceRecognitionService;
import com.viontech.mall.service.adapter.GateService;
import com.viontech.mall.service.adapter.ZoneGateService;
import com.viontech.mall.service.adapter.ZoneService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class MavingLineReportServiceImpl implements MovingLineReportService {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    @Resource
    private ZoneService zoneService;

    @Resource
    private ZoneGateService zoneGateService;

    @Resource
    private GateService gateService;

    @Resource
    private GateReportDataService gateReportDataService;

    @Resource
    private FaceRecognitionService faceRecognitionService;

    /**
     * 动线区域获取的唯一key
     */
    private static final Short KEY_ZONE_MOVING = 4;

    private static final Short KEY_ZONESTATUS_OPEN = 1;

    private static final Short KEY_ZONEGATETYPE_IN = 1;

    private static final Short KEY_FACE_DIREC = 1;

    private static final Short KEY_PERSONTYPE_CUSTOMER = 0;


    @Override
    public Object zoneFaceMovingLine(List<Short> gender,List<Short> ageStage,Date startDate, Date endDate, Map<String, Object> dataMap) {
        //获取商场所有的动线区域
        OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
        Long[] orgIdArr = (Long[]) dataMap.get("ORGID");
        List<Long> orgIds = Arrays.asList(orgIdArr);
        if (orgIds == null){
            String message = LocalMessageUtil.getMessage("Message.orgIdIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        ZoneExample zoneExample = new ZoneExample();
        ZoneExample.Criteria criteria = zoneExample.createCriteria();
        criteria.andTypeEqualTo(KEY_ZONE_MOVING).andStatusEqualTo(KEY_ZONESTATUS_OPEN);
        if (OrgType.mall.toString().equals(orgType.toString())){
            criteria.andMallIdIn(orgIds);
        }else if(OrgType.floor.toString().equals(orgType.toString())){
            criteria.andFloorIdIn(orgIds);
        }else {
            String message = LocalMessageUtil.getMessage("Message.ortTypeIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        if (zones == null || zones.size() <= 0)
            return JsonMessageUtil.getErrorJsonMsg(LocalMessageUtil.getMessage("Message.ortTypeIsNull"));

        //把所有的动线区域放入node中
        Map<Long,Map> nodeMap = new HashMap<>();
        List<Long> zoneIds = new ArrayList<>();
        Map<Long,Zone> zoneMap = new HashMap<>();
        for (Zone zone : zones) {
            Long zoneId = zone.getId();
            zoneMap.put(zoneId,zone);
            zoneIds.add(zoneId);

            Map<String,Object> node = new HashMap<>();
            node.put("zoneId",zoneId);
            node.put("zoneName",zone.getName());
            node.put("x",zone.getX());
            node.put("y",zone.getY());
            node.put("residence",0);
            node.put("traffic",0);
            nodeMap.put(zoneId,node);
        }

        //获取所有监控点和区域的关系,获取所有动线区域对应的监控点（只取正向的关系）
        Map<Long,Long> gateZoneMap = new HashMap<>();
        List<Long> gateIds = new ArrayList<>();
        ZoneGateExample zoneGateExample = new ZoneGateExample();
        zoneGateExample.createCriteria().andZoneIdIn(zoneIds).andTypeEqualTo(KEY_ZONEGATETYPE_IN);
        List<ZoneGate> zoneGateList = zoneGateService.selectByExample(zoneGateExample);
        if (zoneGateList == null || zoneGateList.size()<=0)
            return JsonMessageUtil.getErrorJsonMsg("未找到监控点和区域的对应关系");
        for (ZoneGate zoneGate : zoneGateList) {
            Long gateId = zoneGate.getGateId();
            Long zoneId = zoneGate.getZoneId();
            gateZoneMap.put(gateId,zoneId);
            gateIds.add(gateId);
        }

        Map<Long,Long> zoneGateMap = new HashMap<>();
        zoneGateList.forEach(zoneGate -> {
            zoneGateMap.put(zoneGate.getGateId(),zoneGate.getZoneId());
        });

        //获取所有的人脸动线区域对应监控点的人脸原始数据
        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
        FaceRecognitionExample.Criteria recognitionExampleCriteria = faceRecognitionExample.createCriteria();
        recognitionExampleCriteria.andGateIdIn(gateIds).andCountdateBetween(startDate,endDate).andGenderIn(gender)
                .andAgeIn(ageStage).andDirectionEqualTo(KEY_FACE_DIREC).andPersonTypeEqualTo(KEY_PERSONTYPE_CUSTOMER);
        faceRecognitionExample.setOrderByClause("person_unid,counttime asc");
        List<FaceRecognition> recognitionList = faceRecognitionService.selectByExample(faceRecognitionExample);
        if (recognitionList == null || recognitionList.size() <= 0)
            return  JsonMessageUtil.getErrorJsonMsg("未找到动线区域的人脸抓拍数据");

        //获取人脸去重后的数据，以人脸特征为key，获取所有人的路径轨迹
        Map<String,List<FaceRecognition>> faceRecMap = new HashMap<>();
        for (FaceRecognition faceRecognition : recognitionList) {
            String personUnid = faceRecognition.getPersonUnid();
            List<FaceRecognition> personFaceList = faceRecMap.get(personUnid);

            FaceRecognition lastData = null;
            if (personFaceList == null){
                personFaceList = new ArrayList<>();
            }else if (personFaceList.size() >0){
                lastData = personFaceList.get(personFaceList.size()-1);
            }
            //同一个监控点抓拍的数据，累加轨迹时间，跳过
            if(lastData != null && zoneGateMap.get(lastData.getGateId()).compareTo(zoneGateMap.get(faceRecognition.getGateId()))==0){
                int trackTime = Optional.ofNullable(lastData.getTrackTime()).orElse(0) +Optional.ofNullable(faceRecognition.getTrackTime()).orElse(0)  ;
                lastData.setTrackTime(trackTime);
                continue;
            }
            personFaceList.add(faceRecognition);
            faceRecMap.put(personUnid,personFaceList);
        }

        Map<String,Map> lineMap = new HashMap<>();

        for (Map.Entry<String, List<FaceRecognition>> faceRecEntry : faceRecMap.entrySet()) {
            List<FaceRecognition> personFaces = faceRecEntry.getValue();
            int faceSize = personFaces.size();
            for (int i = 0; i < faceSize; i++) {
                FaceRecognition faceRecognition = personFaces.get(i);
                Long gateId = faceRecognition.getGateId();
                if (gateId == null)
                    continue;
                Long zoneId = gateZoneMap.get(gateId);
                if (zoneId == null)
                    continue;
                //更新node的数据
                Map<String,Object> node = nodeMap.get(zoneId);
                if (node == null)
                    continue;
                Integer traffic = (Integer) node.get("traffic");
                traffic++;
                Integer residence = (Integer) node.get("residence");
                residence+=Optional.ofNullable(faceRecognition.getTrackTime()).orElse(0);
                node.put("traffic",traffic);
                node.put("residence",residence);
                nodeMap.put(zoneId,node);
                //更新line的数据
                if (i == faceSize-1){
                    continue;
                }
                //顾客下一个人脸抓拍的数据
                FaceRecognition nextFaceRec = personFaces.get(i+1);
                Long next_gateId = nextFaceRec.getGateId();
                if (next_gateId == null)
                    continue;
                Long next_zoneId = gateZoneMap.get(next_gateId);
                if (next_zoneId == null)
                    continue;
                String key =""+zoneId+"_"+next_zoneId;
                Map line = lineMap.get(key);
                if (line == null){
                    line = new HashMap();
                    line.put("source",zoneId);
                    line.put("target",next_zoneId);
                    line.put("value",1);
                }else {
                    Integer value = (Integer) line.get("value");
                    value++;
                    line.put("value",value);
                }
                lineMap.put(key,line);
            }
        }

        Map<String,Object> resultMap = new HashMap<>();

        List<Long> showNodes = new ArrayList<>();
        List<Map> LineList = new ArrayList<>();
        List<Long> showZoneIds = (List<Long>) dataMap.get("showZoneIds");
        for (Map.Entry<String, Map> lineMapEntry : lineMap.entrySet()) {
            Map line = lineMapEntry.getValue();
            Long source = Long.parseLong(""+line.get("source"));
            Long target = Long.parseLong(""+line.get("target"));
            if (!showZoneIds.contains(source) && !showZoneIds.contains(target))
                continue;
            showNodes.add(source);
            showNodes.add(target);
            LineList.add(line);
        }
        resultMap.put("line",LineList);


        List<Map> nodeList = new ArrayList<>();
        for (Map.Entry<Long, Map> nodeMapEntry : nodeMap.entrySet()) {
            Long zoneId = nodeMapEntry.getKey();
            if(!showNodes.contains(zoneId))
                continue;
            nodeList.add(nodeMapEntry.getValue());
        }
        resultMap.put("node",nodeList);

        return JsonMessageUtil.getSuccessJsonMsg(resultMap);
    }

    /**
     * 人脸客流动线分析
     * @param gender
     * @param ageStage
     * @param startDate
     * @param endDate
     * @param zoneSize
     * @param dataMap
     * @return
     */
    public Object trafficMavingLine(List<Short> gender,List<Short> ageStage,Date startDate,Date endDate,Integer zoneSize,Map<String,Object> dataMap){
        //获取商场所有的动线区域
        OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
        Long orgId = (Long) dataMap.get("ORGID");
        if (orgId == null){
            String message = LocalMessageUtil.getMessage("Message.orgIdIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        ZoneExample zoneExample = new ZoneExample();
        ZoneExample.Criteria criteria = zoneExample.createCriteria();
        criteria.andTypeEqualTo(KEY_ZONE_MOVING).andStatusEqualTo(KEY_ZONESTATUS_OPEN);
        if (OrgType.mall.toString().equals(orgType.toString())){
            criteria.andMallIdEqualTo(orgId);
        }else if(OrgType.floor.toString().equals(orgType.toString())){
            criteria.andFloorIdEqualTo(orgId);
        }else {
            String message = LocalMessageUtil.getMessage("Message.ortTypeIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        List<String> zoneIds = new ArrayList<>();
        List<Long> zoneIdList = new ArrayList<>();

        Map<Long,Zone> zoneMap = new HashMap<>();
        for (Zone zone : zones) {
            Long zoneId = zone.getId();
            zoneMap.put(zoneId,zone);
            zoneIds.add("-"+zoneId);
            zoneIdList.add(zoneId);
        }
        if (zoneIdList == null || zoneIdList.size()<=0)
            return  JsonMessageUtil.getErrorJsonMsg("未找到动线区域");

        //获取所有监控点和区域的关系,获取所有动线区域对应的监控点（只取正向的关系）
        Map<Long,Long> gateZoneMap = new HashMap<>();
        List<Long> gateIds = new ArrayList<>();
        ZoneGateExample zoneGateExample = new ZoneGateExample();
        zoneGateExample.createCriteria().andZoneIdIn(zoneIdList).andTypeEqualTo(KEY_ZONEGATETYPE_IN);
        List<ZoneGate> zoneGateList = zoneGateService.selectByExample(zoneGateExample);

        for (ZoneGate zoneGate : zoneGateList) {
            Long gateId = zoneGate.getGateId();
            Long zoneId = zoneGate.getZoneId();
            gateZoneMap.put(gateId,zoneId);
            gateIds.add(gateId);
        }
        if (gateIds == null || gateIds.size()<=0)
            return  JsonMessageUtil.getErrorJsonMsg("动线区域未绑定监控点");

        //如果路径经过区域数据量为空，默认选择经过全部区域的路径
        if (zoneSize == null)
            zoneSize = zones.size();

        //获取人数原始数据(gateIds 不能超过1000个)
        if (gateIds.size()>=1000){
            String message = LocalMessageUtil.getMessage("Message.orgNumOutOfLimit");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }


        //获取人脸的原始数据
        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
        faceRecognitionExample.createColumns().hasGateIdColumn().hasTrackTimeColumn().hasPersonUnidColumn();
        FaceRecognitionExample.Criteria recognitionExampleCriteria = faceRecognitionExample.createCriteria();
        recognitionExampleCriteria.andGateIdIn(gateIds).andCountdateBetween(startDate,endDate).andGenderIn(gender)
                .andAgeIn(ageStage).andDirectionEqualTo(KEY_FACE_DIREC).andPersonTypeEqualTo(KEY_PERSONTYPE_CUSTOMER);
        faceRecognitionExample.setOrderByClause("person_unid,counttime asc");
        List<FaceRecognition> recognitionList = faceRecognitionService.selectByExample(faceRecognitionExample);
        //获取人脸去重后的数据，以人脸特征为key，获取所有人的路径轨迹
        Map<String,List<FaceRecognition>> faceRecMap = new HashMap<>();
        for (FaceRecognition faceRecognition : recognitionList) {
            String personUnid = faceRecognition.getPersonUnid();
            List<FaceRecognition> personFaceList = faceRecMap.get(personUnid);

            FaceRecognition lastData = null;
            if (personFaceList == null){
                personFaceList = new ArrayList<>();
            }else if (personFaceList.size() >0){
                lastData = personFaceList.get(personFaceList.size()-1);
            }
            //同一个监控点抓拍的数据，累加轨迹时间，跳过
            if(lastData != null && lastData.getGateId().compareTo(faceRecognition.getGateId())==0){
                lastData.setTrackTime(lastData.getTrackTime()+faceRecognition.getTrackTime());
                continue;
            }
            personFaceList.add(faceRecognition);
            faceRecMap.put(personUnid,personFaceList);
        }
        //顾客的轨迹进行抽象为简单的字符串
        List<String> personLines = new ArrayList<>();
        for (Map.Entry<String, List<FaceRecognition>> map : faceRecMap.entrySet()) {
            List<FaceRecognition> faceRecognitionList = map.getValue();
            String lineStr = "";
            for (FaceRecognition faceRecognition : faceRecognitionList) {
                Long gateId = faceRecognition.getGateId();
                Long zoneId = gateZoneMap.get(gateId);
                lineStr+="-"+zoneId;
            }
            personLines.add(lineStr);
        }
        //获取区域数为n的所有排量的可能顺序
        List<String> allSort = new ArrayList<>();
        String t ="";
        CombinationZoneId(zoneIds, zoneIds.size(), zoneSize, t,allSort);
        //筛选出来Top的的结果
        Map<String,Integer> randMap = new HashMap<>();
        for (String sort : allSort) {
            for (String personLine : personLines) {
                Integer num = randMap.get(sort);
                if (num == null)
                    num = 0;
                if (personLine.contains(sort))
                    num+=1;
                randMap.put(sort,num);
            }
        }
        List<Map.Entry<String, Integer>> entryList = new ArrayList<>(randMap.entrySet());
        Collections.sort(entryList, (o1, o2) -> o2.getValue().compareTo(o1.getValue()));
        //只获取前10的线路,拼接resultMap
        List<Object> resultList = new ArrayList<>();
        for (int i=0;i<10;i++){
            Map<String,Object> resultMap = new HashMap<>();
            if(entryList.size()<=i)
                break;
            Map.Entry<String, Integer> lineMap = entryList.get(i);

            String sortLine = lineMap.getKey().substring(1);
            String[] zoneIdStrs = sortLine.split("-");
            List<Map> nodeList = new ArrayList<>();
            List<Map> linkList = new ArrayList<>();
            String lineName ="";
            for (String zoneId : zoneIdStrs) {
                Zone zone = zoneMap.get(Long.parseLong(zoneId));

                String zoneName = zone.getName();
                lineName+="-"+zoneName;

                Map<String,Object> node = new HashMap<>();
                node.put("id",zone.getId());
                node.put("name", zoneName);
                node.put("x",zone.getX());
                node.put("y",zone.getY());
                nodeList.add(node);

                Map<String,Object> link = new HashMap<>();
                link.put("source",zoneId);
                linkList.add(link);
                if (linkList.size()>1){
                    int index = linkList.size() - 2;
                    Map<String,Object> target_link = linkList.get(index);
                    linkList.remove(index);
                    target_link.put("target",zoneId);
                    linkList.add(index,target_link);
                }

            }
            resultMap.put("name",lineName.substring(1));
            Integer num = lineMap.getValue();
            resultMap.put("size",num);
            resultMap.put("nodes",nodeList);
            resultMap.put("links",linkList);
            resultList.add(resultMap);
        }
        return JsonMessageUtil.getSuccessJsonMsg(resultList);
    }

    /**
     * 获取数组中个数为m的数组的全排列
     * @param zoneIds
     * @param n
     * @param m
     * @param t
     * @param result
     */
    public static void CombinationZoneId(List<String> zoneIds,int n,int m,String t,List<String> result){
        if (m == 0){
           return;
        }else {
            for (int i=zoneIds.size()-n;i<=zoneIds.size()-m;i++){
                 t+=zoneIds.get(i);
                 CombinationZoneId(zoneIds, zoneIds.size() - i - 1, m - 1, t,result);
                if (m == 1){
                    result.add(t);
                 }
                int lastIndex = t.lastIndexOf("-");
                t = t.substring(0,lastIndex);
            }
        }
    }

    /**
     * 顾客轨迹动线
     * @param personUnid
     * @param startDate
     * @param endDate
     * @param dataMap
     * @return
     */
    public Object customerMavingLine(String personUnid,Date startDate,Date endDate, Map<String, Object> dataMap){
        List<Map<String, Object>> resutList = new ArrayList();
        //获取所有监控点的信息
        OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
        Long orgId = (Long) dataMap.get("ORGID");
        if (orgId == null){
            String message = LocalMessageUtil.getMessage("Message.orgIdIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        GateExample gateExample = new GateExample();
        gateExample.createColumns().hasIdColumn();
        GateExample.Criteria criteria = gateExample.createCriteria();
        criteria.andIsHasFaceEqualTo((short)1);
        if (OrgType.mall.toString().equals(orgType.toString())){
            criteria.andMallIdEqualTo(orgId);
        }else if(OrgType.floor.toString().equals(orgType.toString())){
            criteria.andFloorIdEqualTo(orgId);
        }else {
            String message = LocalMessageUtil.getMessage("Message.ortTypeIsNull");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        List<Gate> gates = gateService.selectByExample(gateExample);
        if (gates == null) {
            String message = LocalMessageUtil.getMessage("监控点未设置，请设置监控点信息");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        List<Long> gateIds = gates.stream().map(gate -> gate.getId()).collect(Collectors.toList());
        //获取监控点对应的店铺
        ZoneGateExample zoneGateExample = new ZoneGateExample();
        zoneGateExample.createCriteria().andGateIdIn(gateIds);
        List<ZoneGate> zoneGates = zoneGateService.selectByExample(zoneGateExample);
        if (zoneGates == null || zoneGates.size() <= 0){
            return  JsonMessageUtil.getErrorJsonMsg("区域--监控点对应关系未找到");
        }
        Map<Long,Long> zoneGateMap = new HashMap<>();
        zoneGates.forEach(zoneGate -> {
            zoneGateMap.put(zoneGate.getGateId(),zoneGate.getZoneId());
            //zoneIds.add(zoneGate.getZoneId());
        });
        List<Long> zoneIds = zoneGates.stream().map(zoneGate -> zoneGate.getZoneId()).collect(Collectors.toList());
        //获取符合条件的店铺
        ZoneExample zoneExample = new ZoneExample();
        zoneExample.createColumns().hasIdColumn().hasNameColumn().hasXColumn().hasYColumn();
        zoneExample.createCriteria().andIdIn(zoneIds).andTypeEqualTo((short)4);
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        if (zones == null) {
            String message = LocalMessageUtil.getMessage("未找到符合条件区域");
            return JsonMessageUtil.getErrorJsonMsg(message);
        }
        Map<Long,Zone> zoneMap = new HashMap<>();
        zones.forEach(zone -> zoneMap.put(zone.getId(),zone));
        Map<Long,Gate> gateMap = new HashMap<>();
        gates.forEach(gate -> gateMap.put(gate.getId(),gate));
        //获取人脸的原始数据
        endDate = DateUtil.setDayMaxTime(endDate);
        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
        faceRecognitionExample.createColumns().hasGateIdColumn().hasIdColumn().hasCounttimeColumn().hasTrackTimeColumn().hasFacePicColumn().hasFacePathColumn().hasCountdateColumn().hasChannelSerialnumColumn();
        faceRecognitionExample.createCriteria().andPersonUnidEqualTo(personUnid).andCounttimeGreaterThanOrEqualTo(startDate)
                .andCounttimeLessThanOrEqualTo(endDate).andPersonTypeEqualTo(KEY_PERSONTYPE_CUSTOMER);
        faceRecognitionExample.setOrderByClause("counttime asc");
        List<FaceRecognition> recognitionList = faceRecognitionService.selectByExample(faceRecognitionExample);
        //人脸数据去除连续、重复的数据
        List<FaceRecognition> dealDataList = new ArrayList<>();
        Optional.ofNullable(recognitionList).orElse(new ArrayList<>());
        for (FaceRecognition faceRecognition : recognitionList) {
            //---------------提前过滤
            Long gateId = faceRecognition.getGateId();
            Gate gate = gateMap.get(gateId);
            Zone zone = zoneMap.get(zoneGateMap.get(gateId));
            if (gate == null){
                logger.warn("监控点Id未找到对应的监控点"+gateId);
                continue;
            }
            if(zone == null){
                logger.warn("区域Id未找到对应的区域");
                continue;
            }
            //----------------
            int size = dealDataList.size();
            FaceRecognition lastData = null;
            if (size >0){ //非第一个数据
                lastData = dealDataList.get(size - 1);
            }
           if(lastData != null && zoneGateMap.get(lastData.getGateId()).compareTo(zoneGateMap.get(faceRecognition.getGateId()))==0){
               lastData.setTrackTime(lastData.getTrackTime()+faceRecognition.getTrackTime());
                continue;
           }
           dealDataList.add(faceRecognition);
        }
        //排序，去重后的数据组成规定的格式
        for (int i = 0; i < dealDataList.size(); i++) {
            Map<String,Object> result = new HashMap<>();
            FaceRecognition current_faceRecognition = dealDataList.get(i);
            Long gateId = current_faceRecognition.getGateId();
            //Gate gate = gateMap.get(gateId);
            Zone zone = zoneMap.get(zoneGateMap.get(gateId));
/*            if (gate == null){
                logger.warn("监控点Id未找到对应的监控点"+gateId);
                continue;
            }*/
            result.put("source",zone.getId());
            result.put("sourceName",zone.getName());
            result.put("sourceX",zone.getX());
            result.put("sourceY",zone.getY());
            FaceRecognition recognition = dealDataList.get(i);
            result.put("sourcePath",recognition.getFacePath());
            result.put("sourceChannelSeries",recognition.getChannelSerialnum());
            result.put("sourceCountdate",recognition.getCountdate());
            result.put("sourcePic", recognition.getFacePic());
            result.put("sourceTrackTime", recognition.getTrackTime());

            if (i > 0 && resutList.size() >0){
                Map<String,Object> last_result =  resutList.get(resutList.size() - 1);
                last_result.put("target",zone.getId());
                last_result.put("targetName",zone.getName());
                last_result.put("targetX",zone.getX());
                last_result.put("targetY",zone.getY());
            }
            resutList.add(result);
        }
        return  JsonMessageUtil.getSuccessJsonMsg(resutList);
    }


    /**
     * 客流动线
     * @param floorId
     * @param startDate
     * @param endDate
     * @param dataMap
     * @return
     */
    @Override
    public Object zoneMovingLineData(Long floorId, Date startDate, Date endDate, Map<String, Object> dataMap) {
        Map<String,Object> resultMap = new HashMap<>();
        //根据楼层id获取楼层的动线区域
        ZoneExample zoneExample = new ZoneExample();
        zoneExample.createCriteria().andFloorIdEqualTo(floorId).andTypeEqualTo(KEY_ZONE_MOVING);
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        if (zones == null || zones.size() <= 0)
            return resultMap;
        Map<Long,Zone> zoneMap = new HashMap<>();
        for (Zone zone : zones) {
            Long id = zone.getId();
            zoneMap.put(id,zone);
        }
        //根据动线区域id，获取动线区域和监控点的对应关系数据
        List<Short> typeList = new ArrayList<>();
        typeList.add((short) 1);
        typeList.add((short) 2);
        Set<Long> zoneIds = zoneMap.keySet();
        ZoneGateExample zoneGateExample = new ZoneGateExample();
        zoneGateExample.createCriteria().andZoneIdIn(new ArrayList<>(zoneIds)).andTypeIn(typeList);
        List<ZoneGate> zoneGates = zoneGateService.selectByExample(zoneGateExample);
        if (zoneGates == null)
            return  resultMap;
        //获取所有监控点的客流数据
        Map<Long,List<ZoneGate>> gateId_zoneGateMap = new HashMap<>();//key = gateId
        Map<Long,List<ZoneGate>> zoneId_zoneGateMap = new HashMap<>();//key = zoneId
        Map<Long,List<Long>> zongGateIdMap = new HashMap<>();//key = zoneId
        Set<Long> gateIds = new HashSet<>();
        for (ZoneGate zoneGate : zoneGates) {
            Long gateId = zoneGate.getGateId();
            gateIds.add(gateId);
            Long zoneId = zoneGate.getZoneId();
            //根据监控点id获取监控点--区域对应关系
            List<ZoneGate> gateId_zoneGateList =gateId_zoneGateMap.get(gateId);
            if ( gateId_zoneGateList == null)
                gateId_zoneGateList = new ArrayList<>();
            gateId_zoneGateList.add(zoneGate);
            gateId_zoneGateMap.put(gateId,gateId_zoneGateList);
            //根据区域id获取监控点--区域对应关系
            List<ZoneGate> zoneId_zoneGateList =zoneId_zoneGateMap.get(zoneId);
            if ( zoneId_zoneGateList == null)
                zoneId_zoneGateList = new ArrayList<>();
            zoneId_zoneGateList.add(zoneGate);
            zoneId_zoneGateMap.put(zoneId,zoneId_zoneGateList);
            //根据区域id，获取区域对应的监控点id
            List<Long> zoneGateIds = zongGateIdMap.get(zoneId);
            if (zoneGateIds == null)
                zoneGateIds = new ArrayList<>();
            zoneGateIds.add(gateId);
            zongGateIdMap.put(zoneId,zoneGateIds);
        }
        List<GateDayCountData> gateTraffic = gateReportDataService.getOrQueryGateBetweenDateDayTraffic(gateIds.toArray(new Long[gateIds.size()]), startDate, endDate, dataMap);
        //组装node数据列表
        List<Object> nodeResultList = new ArrayList<>();
        for (Map.Entry<Long, Zone> Entry : zoneMap.entrySet()) {
            Map<String,Object> nodeMap = new HashMap<>();
            Long zoneId = Entry.getKey();
            Zone zone = Entry.getValue();
            nodeMap.put("zondId",zoneId);
            nodeMap.put("zoneName",zone.getName());
            nodeMap.put("x",zone.getX());
            nodeMap.put("y",zone.getY());
            List<Long> gateIdsOfZone = zongGateIdMap.get(zoneId);
            nodeMap.put("gateIds",gateIdsOfZone);

            int zoneInnum = 0,zoneOutnum =0;
            //获取店铺对应多个监控点进入店铺的方向
            List<ZoneGate> zoneGateList = zoneId_zoneGateMap.get(zoneId);
            if (zoneGateList == null)
                continue;
            List<Long> innum_gateIds = new ArrayList<>();
            List<Long> outnum_gateIds = new ArrayList<>();
            for (ZoneGate zoneGate : zoneGateList) {
                Short type = zoneGate.getType();
                Long gateId = zoneGate.getGateId();
                if (Short.compare(type,(short)1) == 0)
                    innum_gateIds.add(gateId);
                else if (Short.compare(type,(short)2) == 0)
                    outnum_gateIds.add(gateId);
            }
            //根据监控点和店铺的方向关系，判断店铺的进客流和出客流
            for (GateDayCountData gateDayCountData : gateTraffic) {
                Long gateId = gateDayCountData.getGateId();
                if (!gateIdsOfZone.contains(gateId))
                    continue;
                if (innum_gateIds.contains(gateId)){
                    zoneInnum += gateDayCountData.getInnum();
                    zoneOutnum += gateDayCountData.getOutnum();
                }else if (outnum_gateIds.contains(gateId)){
                    zoneInnum += gateDayCountData.getOutnum();
                    zoneOutnum += gateDayCountData.getInnum();
                }
            }
            nodeMap.put("innum",zoneInnum);
            nodeMap.put("outnum",zoneOutnum);
            nodeResultList.add(nodeMap);
        }
        resultMap.put("node",nodeResultList);
        //组装line数据列表
        List<Map<String,Object>> lineResultList = new ArrayList<>();
        for (ZoneGate zoneGate : zoneGates) {
            Long id = zoneGate.getId();
            Long zoneId = zoneGate.getZoneId();
            Long gateId = zoneGate.getGateId();
            Short type = zoneGate.getType();

            Map<String,Object> lineMap = new HashMap<>();
            lineMap.put("target",zoneId);
            lineMap.put("value",0);
            for (GateDayCountData gateDayCountData : gateTraffic) {
                Long data_gateId = gateDayCountData.getGateId();
                if (!gateId.equals(data_gateId))
                    continue;
                if (Short.compare(type,(short)1) == 0)
                    lineMap.put("value",gateDayCountData.getInnum());
                else
                    lineMap.put("value",gateDayCountData.getOutnum());
            }

            List<ZoneGate> zoneGateList = gateId_zoneGateMap.get(gateId);
            for (ZoneGate target_zoneGate : zoneGateList) {
                Long target_Id = target_zoneGate.getId();
                Short target_type = target_zoneGate.getType();
                Long target_zoneId = target_zoneGate.getZoneId();
                if (target_Id.equals(id))
                    continue;
                if (type.equals(target_type))
                    continue;
                lineMap.put("source",target_zoneId);
                lineResultList.add(lineMap);
            }
        }
        resultMap.put("line",lineResultList);
        return resultMap;
    }
}
