package com.viontech.mall.report.service.impl;


import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.viontech.keliu.i18n.util.LocalMessageUtil;
import com.viontech.keliu.redis.RedisUtil;
import com.viontech.keliu.storage.Storage;
import com.viontech.keliu.util.DateUtil;
import com.viontech.keliu.util.JsonMessageUtil;
import com.viontech.keliu.util.NumberUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.enums.OptionsContain;
import com.viontech.mall.report.model.ParamModel;
import com.viontech.mall.report.model.Track;
import com.viontech.mall.report.model.TrackInfo;
import com.viontech.mall.report.service.adapter.HeatmapChartService;
import com.viontech.mall.service.adapter.*;
import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import static com.viontech.keliu.i18n.util.LocalMessageUtil.getMessage;

/**
 * 热力图报表
 */
@Service
public class HeatmapChartServiceImpl implements HeatmapChartService {

    Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private ZoneDayCountDataService zoneDayCountDataService;

    @Autowired
    private ZoneService zoneService;

    @Resource
    private GateService gateService;

    @Resource
    private ChannelService channelService;

    @Resource
    private ZoneGateService zoneGateService;

    @Resource
    private GateHourCountDataService gateHourCountDataService;

    @Resource
    private HeatmapDataRawService heatmapDataRawService;

    @Resource
    private FaceRecognitionService faceRecognitionService;

    @Resource
    private Storage simpleStringStorage;


    @Resource
    private ObjectMapper objectMapper;

    /**
     * 根据楼层id获取楼层列表
     *
     * @param floorId
     * @return
     */
    public List<HeatmapData> getGateHourTraffic(Long floorId, Date startDate, Date endDate) {
        List<HeatmapData> datas = new ArrayList<>();
        //根据楼层id获取该楼层的监控点
        GateExample gateExample = new GateExample();
        gateExample.createCriteria().andFloorIdEqualTo(floorId);
        List<Gate> gates = gateService.selectByExample(gateExample);
        if (gates == null)
            return datas;
        //获取楼层所有监控点的小时数据
        List<Long> gateIds = gates.stream().map(Gate::getId).collect(Collectors.toList());
        if (gateIds == null || gateIds.size() <= 0)
            return datas;

        GateHourCountDataExample gateHourCountDataExample = new GateHourCountDataExample();
        gateHourCountDataExample.createColumns().hasHourColumn().hasInnumColumn().hasOutnumColumn().hasCountdateColumn().hasCounttimeColumn().hasGateIdColumn();
        gateHourCountDataExample.createGateColumns().hasXColumn().hasYColumn().hasNameColumn().hasTypeColumn();
        gateHourCountDataExample.createCriteria().andGateIdIn(gateIds).andCounttimeGreaterThanOrEqualTo(startDate).andCounttimeLessThanOrEqualTo(endDate);

        String tableAlias = gateHourCountDataExample.getTableAlias();
        gateHourCountDataExample.setOrderByClause(tableAlias + ".gate_id," + tableAlias + ".counttime");

        List<GateHourCountData> gateHourDatas = gateHourCountDataService.selectByExample(gateHourCountDataExample);
        Map<Long, HeatmapData> heatmapDataMap = new HashMap<>();
        for (GateHourCountData data : gateHourDatas) {
            Long gateId = data.getGateId();
            HeatmapData heatmapData = heatmapDataMap.get(gateId);
            if (heatmapData == null) {
                heatmapData = new HeatmapData();
                heatmapData.setGateId(gateId);
                heatmapData.setFloorId(floorId);
                Gate gate = data.getGate();
                if (gate == null)
                    continue;
                heatmapData.setX(gate.getX());
                heatmapData.setY(gate.getY());
                heatmapData.setCounttime(data.getCounttime());
                heatmapData.setCountnum(0);
            }
            Integer innum = data.getInnum() + heatmapData.getCountnum();
            heatmapData.setCountnum(innum);
            heatmapDataMap.put(gateId, heatmapData);
        }
        List<HeatmapData> result = new ArrayList<>();
        for (HeatmapData heatmapData : heatmapDataMap.values()) {
            result.add(heatmapData);
        }
        return result;
    }


    /**
     * 根据设备序列号获取热力图原始数据
     * @param channelSerialnum
     * @param startDate
     * @param endDate
     * @return
     */
    public List<HeatmapDataRaw> getHeatmapDataBySerialnum(String channelSerialnum, Date startDate, Date endDate){
        List<HeatmapDataRaw> heatmapDatas = new ArrayList<>();
       HeatmapDataRawExample heatmapDataRawExample = new HeatmapDataRawExample();
       heatmapDataRawExample.createCriteria().andChannelSerialnumEqualTo(channelSerialnum).andCounttimeGreaterThanOrEqualTo(startDate)
              .andCounttimeLessThanOrEqualTo(endDate);
       heatmapDatas = heatmapDataRawService.selectByExample(heatmapDataRawExample);
       return  heatmapDatas;
    }

    /**
     * 根据人脸数据获取区域的热力图（连锁客流）
     * @param orgId   商场id
     * @param startDate  开始时间
     * @param endDate  介绍时间
     * @param tab  热力图展示方式  innum  人数  residence  滞留时间   average  平均滞留时间
     * @return
     */
    public Object getZoneHotmapByFaceForMall(Long orgId,Date startDate,Date endDate,String tab){
        Map<Long,Map<String,Object>> trackMap = new HashMap<>();
        //获取广场所有的人脸监控点
        GateExample gateExample = new GateExample();
        gateExample.createCriteria().andMallIdEqualTo(orgId).andIsHasFaceEqualTo((short) 1).andStatusEqualTo((short) 1);
        List<Gate> gates = gateService.selectByExample(gateExample);
        if (gates == null || gates.size() <= 0){
          return  JsonMessageUtil.getErrorJsonMsg("广场未找到对应的人脸相机");
        }
        List<Long> gateIds = gates.stream().map(gate -> gate.getId()).collect(Collectors.toList());
        //获取监控点对应的店铺
        ZoneGateExample zoneGateExample = new ZoneGateExample();
        zoneGateExample.createCriteria().andGateIdIn(gateIds);
        List<ZoneGate> zoneGates = zoneGateService.selectByExample(zoneGateExample);
        if (zoneGates == null || zoneGates.size() <= 0){
            return  JsonMessageUtil.getErrorJsonMsg("区域--监控点对应关系未找到");
        }
        List<Long> zoneIds = new ArrayList();
        Map<Long,Long> zoneGateMap = new HashMap<>();
        zoneGates.forEach(zoneGate -> {
            zoneGateMap.put(zoneGate.getGateId(),zoneGate.getZoneId());
            zoneIds.add(zoneGate.getZoneId());
        });
        //获取监控点对应的店铺信息
        ZoneExample zoneExample = new ZoneExample();
        zoneExample.createCriteria().andMallIdEqualTo(orgId).andStatusEqualTo((short) 1).andIdIn(zoneIds);
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        if (zones == null || zones.size() <= 0){
            return  JsonMessageUtil.getSuccessJsonMsg("广场未找到对应的区域");
        }
        //把所有的区域先放入结果集合中
        for (Zone zone : zones) {
            Map<String,Object> result = new HashMap<>();
            Long zoneId = zone.getId();
            result.put("zoneName",zone.getName());
            result.put("zoneX",zone.getX());
            result.put("zoneY",zone.getY());
            trackMap.put(zoneId,result);
        }

        //获取广场所有的人脸抓拍数据
        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
        faceRecognitionExample.createColumns().hasGateIdColumn().addColumnStr("sum(1) as count , sum(track_time) as faceRecognition_track_time ");

        faceRecognitionExample.createCriteria().andMallIdEqualTo(orgId).andCountdateGreaterThanOrEqualTo(startDate)
                .andCountdateLessThanOrEqualTo(endDate);
        faceRecognitionExample.setGroupByClause("gate_id,track_time");
        faceRecognitionExample.setOrderByClause("gate_id");
        List<FaceRecognition> recognitions = faceRecognitionService.selectByExample(faceRecognitionExample);
        //根据tab把数据放入每个监控点
        for (FaceRecognition recognition : recognitions) {
            Long gateId = recognition.getGateId();
            Long zoneId = zoneGateMap.get(gateId);
            if (zoneId == null){
               // logger.error("监控点{}未找到对应的区域",gateId);
                continue;
            }
            Map<String, Object> result = trackMap.get(zoneId);
            if ("innum".equals(tab)){
                int innum = (int) Optional.ofNullable(result.get("innum")).orElse(0);
                //innum++;
                innum+=recognition.getCount();
                result.put("innum",innum);
            }
            //Integer trackTime = recognition.getTrackTime();
            //Integer times = (Integer) Optional.ofNullable(result.get("residence")).orElse(0);
            //Number residence = (Number) NumberUtil.valueAdd(times, trackTime);
            if ("residence".equals(tab)) {
                int residence = (int)Optional.ofNullable(result.get("residence")).orElse(0);
                System.out.println("滞留时间"+recognition.getTrackTime());
                int nowResidence = (int)Optional.ofNullable(recognition.getTrackTime()).orElse(0);
                residence+=nowResidence;
                result.put("residence", residence);
            }
            if ("average".equals(tab)) {
                int innum = (int) Optional.ofNullable(result.get("innum")).orElse(0);
                //innum++;
                innum+=recognition.getCount();
                int residence = (int)Optional.ofNullable(result.get("residence")).orElse(0);
                System.out.println("滞留时间"+recognition.getTrackTime());
                int nowResidence = (int)Optional.ofNullable(recognition.getTrackTime()).orElse(0);
                residence+=nowResidence;
                result.put("average", NumberUtil.divide(residence, innum, 2));
            }
            trackMap.put(zoneId,result);
        }
        return JsonMessageUtil.getSuccessJsonMsg(trackMap);
    }

    /**
     * 店铺人脸轨迹热力图（连锁客流）
     * @return
     */
//    @Override
//   public Object trackHeatmap(Long channelId, Date startDate, Date endDate) {
//        /*//获取店铺的所有人脸相机
//        ChannelExample channelExample = new ChannelExample();
//        channelExample.createCriteria().andMallIdEqualTo(orgId).andTypeEqualTo((short) 1).andStatusEqualTo((short) 1);
//        List<Channel> channels = channelService.selectByExample(channelExample);
//        List<Long> faceChannelIds = channels.stream().map(channel -> channel.getId()).collect(Collectors.toList());*/
//        //获取所有人脸相机的筛选数据
//        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
//        faceRecognitionExample.createCriteria().andCountdateBetween(startDate,endDate).andChannelIdEqualTo(channelId);
//        faceRecognitionExample.createColumns().hasTrackPathColumn().hasTrackInfoColumn().hasChannelSerialnumColumn();
//        List<FaceRecognition> faceRecognitions = faceRecognitionService.selectByExample(faceRecognitionExample);
//        //通过人脸数据获取轨迹坐标数据，归属到对应的通道
//        Map<String,List> trackMap = new ConcurrentHashMap<>();
//        //创建线程池
//       /* BlockingQueue<Runnable> queue = new  ArrayBlockingQueue<>(10);
//        ExecutorService executor  = new ThreadPoolExecutor(
//                5,      //core
//                10,     //max
//                120L,   //2分钟
//                TimeUnit.SECONDS,
//                queue);*/
//
//        long startTime = System.currentTimeMillis();
//        System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>>>>轨迹文件数目为："+ faceRecognitions.size());
//        System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>>>>开始读取轨迹文件："+ startTime);
//        /*for (FaceRecognition faceRecognition : faceRecognitions) {
//                executor.submit(() -> {
//                    dealFaceDataToTrack(trackMap, faceRecognition);
//                });
//        }
//        executor.shutdown();*/
//        //faceRecognitions.parallelStream().forEach(faceRecognition -> { dealFaceDataToTrack(trackMap, faceRecognition);});
//
//        for (FaceRecognition faceRecognition : faceRecognitions) {
//                dealFaceDataToTrack(trackMap, faceRecognition);
//        }
//
//        System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>>>>读取轨迹文件结束，消耗时间："+(System.currentTimeMillis()-startTime)*0.001);
//        //对同一个通道的相同坐标进行聚合，热力值相加
//        Map<String,List> resultMap = new HashMap<>();
//        for (Map.Entry<String, List> entry : trackMap.entrySet()) {
//            String channelSeriesNum = entry.getKey();
//            List tracks = entry.getValue();
//            Map<String,Track> trackAddMap = new HashMap<>();
//            for (Object o : tracks) {
//                Track track = (Track) o;
//                String key = ""+track.getX()+","+track.getY();
//                Track trackFromMap = trackAddMap.get(key);
//                if (trackFromMap == null)
//                    trackFromMap = track;
//                else{
//                    Integer count_trackFromMap = Optional.ofNullable(trackFromMap.getCount()).orElse(0);
//                    Integer count_track = Optional.ofNullable(track.getCount()).orElse(0);
//                    trackFromMap.setCount(count_trackFromMap+count_track);
//                }
//                trackAddMap.put(key,trackFromMap);
//            }
//            List<Track> resultTrack = trackAddMap.values().stream().collect(Collectors.toList());
//            resultMap.put(channelSeriesNum,resultTrack);
//        }
//        return JsonMessageUtil.getSuccessJsonMsg(resultMap);
//    }


    private String getTrackInfoStr(FaceRecognition faceRecognition) {
        //获取抓拍记录对应的轨迹文件
        String trackPath = faceRecognition.getTrackPath()+""+faceRecognition.getTrackInfo();
        Object collectData = simpleStringStorage.getItem(trackPath);
        if (collectData == null) {
            logger.info("轨迹数据获取失败+"+trackPath);
            return null;
        }
        else{
            return collectData.toString();
        }

    }

    /**
     * 楼层店铺相对温度和纵向温度
     *
     * @param floorId
     * @param paramModel
     * @return
     */
    @Override
    public Object getFloorHeatmap(Long floorId,ParamModel paramModel) {
        Map<Long, Object> resultMap = new HashMap<>();

        List<Short> zoneTypes = new ArrayList<Short>();
        zoneTypes.add((short)1);
        zoneTypes.add((short)3);
        ZoneExample zoneExample = new ZoneExample();
        zoneExample.createColumns().hasIdColumn();
        //TODO   .andTypeIn(zoneTypes)  放入过滤条件中
        zoneExample.createCriteria().andFloorIdEqualTo(floorId).andTypeIn(zoneTypes).andStatusEqualTo((short) 1);
        List<Zone> zones = zoneService.selectByExample(zoneExample);
        if (zones == null || zones.size() <= 0)
            return JsonMessageUtil.getErrorJsonMsg(getMessage("Message.OrgNotFind"));
        //符合条件的店铺id(去重)
        Set<Long> zonesIds = zones.stream().map(Zone::getId).collect(Collectors.toSet());

        //判断请求是相对温度或纵向温度
        OptionsContain option = paramModel.getOption();
        if (option == null)
            return JsonMessageUtil.getErrorJsonMsg(getMessage("Message.paramsIsError"));

        Map<String,Object> mapTemple = null;
        if (OptionsContain.TAB_RELATIVETEMPERATURE.equals(option)){//相对温度   获取时间段
            Date startDate = paramModel.getStartDate() == null ? new Date():paramModel.getStartDate();
            Date endDate = paramModel.getEndDate() == null ? new Date():paramModel.getEndDate();
            Map<Long, Object> relativeTemperatureMap = getRelativeTemperature(floorId, startDate,endDate,zonesIds);
            for (Long key : zonesIds) {
                mapTemple = new HashMap();
                if(relativeTemperatureMap != null) {
                    mapTemple.put("relativeTemperature", relativeTemperatureMap.get(key));
                }
                resultMap.put(key,mapTemple);
            }
            //把最大最小值放入结果集
            resultMap.put(-2L,relativeTemperatureMap.get(-2L));
            resultMap.put(-1L,relativeTemperatureMap.get(-1L));
            resultMap.put(-3L,relativeTemperatureMap.get(-3L));
        }else {//纵向温度   获取时间点
            Date date = paramModel.getDate() == null ? new Date():paramModel.getDate();
            Map<Long, Object> portraitTemperatureMap = getPortraitTemperature(floorId, date,zonesIds);
            for (Long key : zonesIds) {
                mapTemple = new HashMap();
                if(portraitTemperatureMap != null) {
                    mapTemple.put("portraitTemperature", portraitTemperatureMap.get(key));
                }
                resultMap.put(key,mapTemple);
            }
        }
        return JsonMessageUtil.getSuccessJsonMsg(resultMap);
    }

    /**
     * 获取某个楼层的相对温度
     */
    public Map<Long, Object> getRelativeTemperature(Long floorId,Date startDate,Date endDate,Set<Long> zonesIds) {
        ZoneDayCountDataExample zoneDayCountDataExample = new ZoneDayCountDataExample();
        zoneDayCountDataExample.createColumns().hasZoneIdColumn().addColumnStr("sum(innum) as zoneDay_innum");
        zoneDayCountDataExample.createCriteria().andCountdateBetween(startDate,endDate).andFloorIdEqualTo(floorId).andInnumIsNotNull().andZoneIdIn(new ArrayList<>(zonesIds));
        zoneDayCountDataExample.setOrderByClause(zoneDayCountDataExample.getTableAlias()+".zone_id");
        zoneDayCountDataExample.setGroupByClause("zone_id");
        List<ZoneDayCountData> zoneDayCountDataList = zoneDayCountDataService.selectByExample(zoneDayCountDataExample);
        if (zoneDayCountDataList == null) {
            return null;
        }
        Map<Long, Object> relativeTemperatureMap = new HashMap<>();

        //int min = Integer.MAX_VALUE;
        //int max = Integer.MIN_VALUE;
        Collections.sort(zoneDayCountDataList, new Comparator<ZoneDayCountData>() {
            @Override
            public int compare(ZoneDayCountData o1, ZoneDayCountData o2) {
                return o1.getInnum().compareTo(o2.getInnum());
            }
        });
        int length = zoneDayCountDataList.size();
        int min=zoneDayCountDataList.get(0).getInnum();
        int max=zoneDayCountDataList.get(length-1).getInnum();
        int downIndex = (int)Math.floor(length*0.25);
        int upIndex = (int)Math.floor(length*0.75);
        int exceptionMax = (int)Math.round(zoneDayCountDataList.get(upIndex).getInnum()+(zoneDayCountDataList.get(upIndex).getInnum()-zoneDayCountDataList.get(downIndex).getInnum())*1.5);
        if(min == max) {
            return null;
        }
        for (ZoneDayCountData zoneDayCountData : zoneDayCountDataList) {
            List<Integer> list = new ArrayList<Integer>();
            //System.out.println(">>>>>>>>>>>>>>>> min:"+min+"\tmax:"+max+"\tX:" +zoneDayCountData.getInnum() +"\t level:"+(int) Math.floor((zoneDayCountData.getInnum() - min) / ((max - min) / 4.0)+1));
            if(max<exceptionMax) {
                list.add(0, (int) Math.round((zoneDayCountData.getInnum() - min) / ((max - min) / 4.0) + 1));
            }else {
                if(zoneDayCountData.getInnum()>exceptionMax){
                    list.add(0, 5);
                }else {
                    list.add(0, (int) Math.round((zoneDayCountData.getInnum() - min) / ((exceptionMax - min) / 3.0) + 1));
                }
            }
            list.add(1,zoneDayCountData.getInnum());
            relativeTemperatureMap.put(zoneDayCountData.getZoneId(),list);
        }
        relativeTemperatureMap.put(-2L,min);
        relativeTemperatureMap.put(-1L,max);
        relativeTemperatureMap.put(-3L,exceptionMax);
        return relativeTemperatureMap;
    }

    /**
     * 获取某个楼层的纵向温度
     */
    public Map<Long, Object> getPortraitTemperature(Long floorId, Date countDate,Set<Long> zonesIds) {
        Map<Long, Object> portraitTemperatureMap = new HashMap<>();
        List<Date> dateList = new ArrayList<>();
        dateList.add(countDate);
        dateList.add(DateUtil.addDays(countDate,-7));
        dateList.add(DateUtil.addDays(countDate,-14));
        dateList.add(DateUtil.addDays(countDate,-21));
        dateList.add(DateUtil.addDays(countDate,-28));
        dateList.add(DateUtil.addDays(countDate,-35));

        ZoneDayCountDataExample zoneDayCountDataExample = new ZoneDayCountDataExample();
        zoneDayCountDataExample.createCriteria().andFloorIdEqualTo(floorId).andCountdateIn(dateList).andZoneIdIn(new ArrayList<>(zonesIds));
        zoneDayCountDataExample.createColumns().hasInnumColumn().hasZoneIdColumn().hasCountdateColumn();
        zoneDayCountDataExample.setOrderByClause("\"zoneDay\".zone_id,\"zoneDay\".countdate");
        List<ZoneDayCountData> zoneDayCountDataList = zoneDayCountDataService.selectByExample(zoneDayCountDataExample);
        if (zoneDayCountDataList == null || zoneDayCountDataList.isEmpty()) {
            return null;
        }
        Long zoneId = null;
        List<ZoneDayCountData> zoneDayList = null;
        for (ZoneDayCountData zoneDayCountData : zoneDayCountDataList) {
            if(zoneId == null || zoneId.longValue() != zoneDayCountData.getZoneId().longValue()){
                if(zoneDayList==null){
                    zoneDayList = new ArrayList<>();
                }else{
                    int min=Integer.MAX_VALUE;
                    int max = Integer.MIN_VALUE;
                    Integer todayInnum = null;
                    Long currentZoneId = null;
                    for (ZoneDayCountData dayCountData : zoneDayList) {
                        int innum = dayCountData.getInnum();
                        if(DateUtil.isSameDay(dayCountData.getCountdate(),countDate)){
                            todayInnum= innum;
                            currentZoneId = dayCountData.getZoneId();
                        }else{
                            if(min>innum){
                                min = innum;
                            }
                            if(max<innum){
                                max = innum;
                            }
                        }
                    }
                    if(currentZoneId != null){
                        List<Integer> list = new ArrayList<>();
                        list.add(0,getPortraitTemperatureLeval(max, min, todayInnum));
                        list.add(1,todayInnum);
                        portraitTemperatureMap.put(currentZoneId,list);
                    }
                    zoneDayList.clear();
                }
                zoneId = zoneDayCountData.getZoneId().longValue();
            }

            zoneDayList.add(zoneDayCountData);
        }
        return portraitTemperatureMap;
    }

    /**
     * 获取纵向温度等级
     */
    public int getPortraitTemperatureLeval(int max, int min, int traffic) {
        if (traffic == 0 || traffic <= min) {
            return 1;
        } else if (traffic > min && traffic <= ((max - min) / 3 + min)) {
            return 2;
        } else if (traffic > ((max - min) / 3 + min) && traffic <= (max - min) / 3 * 2 + min) {
            return 3;
        } else if (traffic > (max - min) / 3 * 2 + min && traffic <= (max - min) + min) {
            return 4;
        } else {
            return 5;
        }
    }

    /**
     * 店铺人脸轨迹热力图（连锁客流）redis版本
     * @param channelId
     * @param startDate
     * @param endDate
     * @return
     */
    @Override
    public Object trackHeatmapByRedis(Long channelId, Date startDate, Date endDate) {
        ChannelExample channelExample = new ChannelExample();
        channelExample.createCriteria().andIdEqualTo(channelId);
        List<Channel> channels = channelService.selectByExample(channelExample);

        if (channels == null || channels.size() <= 0)
            return JsonMessageUtil.getErrorJsonMsg(LocalMessageUtil.getMessage("Message.channelIdCantFind"));

        Map<String,List> resultMap = new HashMap<>();
        String channelSerialnum = channels.get(0).getSerialnum();

        TIntObjectHashMap<TIntIntHashMap> collectMap = new TIntObjectHashMap();
        List<Date> daysBetweenDates = DateUtil.getDaysBetweenDates(startDate, endDate);//返回多天日期
        for (Date currentDate : daysBetweenDates) {
            String yyyyMMdd = DateUtil.format("yyyyMMdd", currentDate);
            //获取获取轨迹汇总文件
            List<TrackInfo> trackInfos = new ArrayList<>();
            Object collectData = simpleStringStorage.getItem("trackgroup/" + yyyyMMdd + "/" + channelSerialnum + ".track");
            //未找到汇总文件时，从抓拍记录中找到碎片轨迹文件，进行汇总
            if (collectData == null) {
                if (DateUtil.isSameDay(currentDate, new Date())) {//当天的数据从redis中获取
                    trackInfos = getTrackInfosFromReids(channelSerialnum, currentDate);
                } else {//非当天从抓拍记录表中获取所有抓拍的轨迹文件
                    logger.info("开始获取抓拍中的轨迹数据");
                    trackInfos =  getTrackInfosFromDB(channelId, currentDate);
                }
                if (!DateUtil.isSameDay(currentDate, new Date())){//非当天的数据汇总后进行存储
                    String trackInfoJson = "";
                    try {
                        trackInfoJson = objectMapper.writeValueAsString(trackInfos);
                        logger.debug("日期:{},通道:{},轨迹文件存储完成，汇总数量为:{}", yyyyMMdd, channelSerialnum, trackInfos.size());
                    } catch (JsonProcessingException e) {
                        e.printStackTrace();
                    }
                    String trackKey = "trackgroup/" + yyyyMMdd + "/" + channelSerialnum + ".track";
                    simpleStringStorage.setItem(trackKey, trackInfoJson);
                }
            }else{
                JavaType javaType = objectMapper.getTypeFactory().constructParametricType(ArrayList.class, TrackInfo.class);
                try {
                    trackInfos = objectMapper.readValue(collectData.toString(), javaType);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (trackInfos.size() <= 0)
                continue;

            //循环获取到的轨迹数据，轨迹数据进行汇总
            for (TrackInfo trackInfo : trackInfos) {
                List<Track> tracks = trackInfo.getTrack();
                for (Track track : tracks) {
                    //对轨迹数据进行压缩、汇总
                    compressTrackData(collectMap, track);
                }
            }
        }
        //格式化返回对象
        List<Map<String,Integer>> trackList = new ArrayList<>();
        collectMap.forEachEntry((x, tIntIntHashMap) -> {
            tIntIntHashMap.forEachEntry((y, v) -> {
                Map<String,Integer> trackMap = new HashMap<>();
                trackMap.put("x",x);//key
                trackMap.put("y",y);
                trackMap.put("v",v);
                trackList.add(trackMap);
                return true;
            });
            return true;
        });
        resultMap.put(channelSerialnum,trackList);
        return JsonMessageUtil.getSuccessJsonMsg(resultMap);
    }

    /**
     * 压缩轨迹数据
     * @param collectMap  压缩后轨迹数据存放的数据容器
     * @param track  当前需要压缩的轨迹对象
     */
    private void compressTrackData(TIntObjectHashMap<TIntIntHashMap> collectMap, Track track) {
        Integer x = track.getX();
        if (x % 8 == 0) {
            x = x + 4;
        } else {
            x = x - x % 8 + 4;
        }
        TIntIntHashMap yObject =  collectMap.get(x);
        if (yObject == null){
            yObject =new TIntIntHashMap();
        }
        Integer y = track.getY();
        if (y % 8 == 0) {
            y = y + 4;
        } else {
            y = y - y % 8 + 4;
        }
        Integer count =Optional.ofNullable(yObject.get(y)).orElse(0);
        if (count == null){
            count = 0;
        }
        count++;
        yObject.put(y,count);
        collectMap.put(x,yObject);
    }

    /**
     * 历史轨迹数据从人脸抓拍表中进行获取
     * @param channelId
     * @param currentDate
     * @return
     */
    private List<TrackInfo> getTrackInfosFromDB(Long channelId, Date currentDate) {
        List<TrackInfo> trackInfos = new ArrayList<>();
        FaceRecognitionExample faceRecognitionExample = new FaceRecognitionExample();
        faceRecognitionExample.createCriteria().andCountdateEqualTo(currentDate).andChannelIdEqualTo(channelId);
        faceRecognitionExample.createColumns().hasTrackPathColumn().hasTrackInfoColumn().hasChannelSerialnumColumn();
        List<FaceRecognition> faceRecognitions = faceRecognitionService.selectByExample(faceRecognitionExample);
        int size = faceRecognitions.size();
        int index = 1;
        for (FaceRecognition faceRecognition : faceRecognitions) {
            String trackPath = faceRecognition.getTrackPath();
            String trackName = faceRecognition.getTrackInfo();
            Object trackObj = simpleStringStorage.getItem(trackPath + trackName);
            if (trackObj == null) {
                logger.warn("{}的轨迹文件信息无法获取到", trackName);
                continue;
            }
            logger.debug("通道:{}估计汇总,总条数:{},当前处理条数:{}", ""+faceRecognition.getChannelSerialnum(), size, index++);
            try {
                TrackInfo trackInfo = objectMapper.readValue(trackObj.toString(), TrackInfo.class);
                trackInfos.add(trackInfo);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return trackInfos;
    }


    /**
     * 当天轨迹数据从redis中获取
     * @param channelSerialnum
     * @param currentDate
     * @return
     */
    private List<TrackInfo> getTrackInfosFromReids(String channelSerialnum, Date currentDate) {
        List<TrackInfo> trackInfos = new ArrayList<>();
        String format = DateUtil.format(DateUtil.FORMAT_SHORT, currentDate);
        String keys = "track:" + format + ":" + channelSerialnum;
        Set<Object> zSetRange = RedisUtil.getZSetRange(keys);
        if(zSetRange == null || zSetRange.size() <= 0)
            return trackInfos;
        for (Object obj : zSetRange) {
            if (obj == null) {
                continue;
            }
            TrackInfo trackInfo = null;
            try {
                trackInfo = objectMapper.readValue(obj.toString(), TrackInfo.class);
            } catch (IOException e) {
                e.printStackTrace();
            }
            if(trackInfo!=null) {
                trackInfos.add(trackInfo);
            }
        }
        return trackInfos;
    }

}
