package com.viontech.mall.report.service.impl;

import com.viontech.keliu.chart.Chart;
import com.viontech.keliu.chart.Table;
import com.viontech.keliu.chart.axis.Axis;
import com.viontech.keliu.chart.axis.TableHead;
import com.viontech.keliu.chart.factory.AxisFactory;
import com.viontech.keliu.chart.series.SeriesType;
import com.viontech.keliu.i18n.util.LocalMessageUtil;
import com.viontech.keliu.util.NumberUtil;
import com.viontech.mall.model.*;
import com.viontech.mall.report.base.ChartReportBaseService;
import com.viontech.mall.report.enums.OrgType;
import com.viontech.mall.report.util.AgeProcessUtil;
import com.viontech.mall.service.adapter.GateDayFaceRecognitionStaService;
import com.viontech.mall.service.adapter.GateService;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * 监控点客流分布分析报表
 * @author dugw
 *
 */
@Service
public class FaceDistributionReportServiceImp extends ChartReportBaseService {

	@Value("${Age.stage}")
	private String ageStage;
	
	@Resource
	private GateDayFaceRecognitionStaService gateDayFaceRecognitionStaService;
	
	@Resource
	private GateService gateService;
	
	
	private final static String FELID_GATE_HAS_FACE="gatesHasFace";
	private final static String FELID_FACE_DATA="faceDatas";
	private final static String PARAM_DIC_GENDER = "gender";

	/**
	 * 监控点人脸客流统计
	 */
	private final static  String  REPORT_FACE_TRAFFIC= "faceTraffic";
	/**
	 * 监控点人脸客流统计详情
	 */
	private final static  String  REPORT_FACE_TRAFFIC_DETAIL = "faceTrafficDetail";
	/**
	 * 监控点人脸性别比例统计
	 */
	private final static  String  REPORT_FACE_GENGER_RATIO = "faceGenderRatio";
	/**
	 * 监控点人脸年龄比例统计
	 */
	private final static  String  REPORT_FACE_AGE_RATIO = "faceAgeRatio";
	
	@Override
	public Chart getChart(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap, ReportChart reportChart) {
		//得到需要获取数据的监控点
		Long[] gateIds = getGateHasFace(orgIds, dataMap);
		if(gateIds == null || gateIds.length == 0)
			return null;
		List<GateDayFaceRecognitionSta> faceDatas = getGateDayFaceDatas(gateIds, startDate, endDate, dataMap);

		Chart chart = null;
		switch (reportChart.getKey()) {
		case REPORT_FACE_TRAFFIC:
			chart = faceTrafficReport(gateIds, faceDatas, dataMap, reportChart);
			break;
		case REPORT_FACE_TRAFFIC_DETAIL:
			chart = faceTrafficDatailReport(gateIds, faceDatas, dataMap, reportChart);
			break;
		case REPORT_FACE_GENGER_RATIO:
			chart = faceGenderRatioReport(orgIds, faceDatas, dataMap, reportChart);
			break;
		case REPORT_FACE_AGE_RATIO:
			chart = faceAgeRatioReport(orgIds, faceDatas, dataMap, reportChart);
			break;
		default:
			break;
		}
		return chart;
	}

	/**
	 * 监控点人脸客流统计详情
	 * @param orgIds
	 * @param faceDatas
	 * @param dataMap
	 * @param reportChart
	 * @return
	 */
	private Chart faceTrafficDatailReport(Long[] orgIds,List<GateDayFaceRecognitionSta> faceDatas,Map<String, Object> dataMap, ReportChart reportChart){
		final String gateName = LocalMessageUtil.getMessage("Face.gateName");
		final String costomerTime = LocalMessageUtil.getMessage("Face.customerTime");
		final String customerCount = LocalMessageUtil.getMessage("Face.customerCount");
		final String male = LocalMessageUtil.getMessage("ParamName.male");
		final String female = LocalMessageUtil.getMessage("ParamName.female");
		final String unknown = LocalMessageUtil.getMessage("ParamName.unknown");

		Table table = new Table(reportChart.getTitle());
		TableHead tableHead = new TableHead();
		tableHead.addData(gateName);
		tableHead.addData(costomerTime);
		tableHead.addData(customerCount);
		tableHead.addData(male);
		tableHead.addData(female);
		tableHead.addData(unknown);
		table.setTableHead(tableHead);

		//人脸监控点放入表格第一列，（预防该监控点无数据时，监控点不显示的问题）
		Map<Long, Gate> gatesHasFace = getGatesHasFace(orgIds, dataMap);
		for (Gate gate : gatesHasFace.values()) {
			table.getRow(gate.getName()).putValueByHeadColumn(gateName, gate.getName());
		}
		
		//人脸数据
		for (GateDayFaceRecognitionSta face : faceDatas) {
			Long gateId = face.getGateId();
			Gate gate = gatesHasFace.get(gateId);
			table.getRow(gate.getName()).putValueByHeadColumn(gateName, gate.getName());
			table.getRow(gate.getName()).adjustOrPutValueByHeadColumn(costomerTime, face.getCustomMantime());
			table.getRow(gate.getName()).adjustOrPutValueByHeadColumn(customerCount,face.getCustomCount());
			table.getRow(gate.getName()).adjustOrPutValueByHeadColumn(male,face.getMaleCount());
			table.getRow(gate.getName()).adjustOrPutValueByHeadColumn(female,face.getFemaleCount());
			table.getRow(gate.getName()).adjustOrPutValueByHeadColumn(unknown,face.getCustomCount()-face.getMaleCount()-face.getFemaleCount());
		}
		return table;
	}
	
	/**
	 * 监控点人脸客流统计
	 * @param orgIds
	 * @param faceDatas
	 * @param dataMap
	 * @param reportChart
	 * @return
	 */
	private Chart faceTrafficReport(Long[] orgIds,List<GateDayFaceRecognitionSta> faceDatas,Map<String, Object> dataMap, ReportChart reportChart){
		Chart chart = new Chart(reportChart.getTitle(), SeriesType.bar);
		
		Axis<String> xAxis = AxisFactory.createStringAxis();
		Map<Long, Gate> gatesHasFace = getGatesHasFace(orgIds, dataMap);
		for (Gate gate : gatesHasFace.values()) {
			xAxis.addData(gate.getName());
		}
		chart.setXAxis(xAxis);
		
		final String manTime = LocalMessageUtil.getMessage("Face.allTime");
		final String manCount = LocalMessageUtil.getMessage("Face.allCount");
		
		//人脸数据faceDatas
		for (GateDayFaceRecognitionSta face : faceDatas) {
			Long gateId = face.getGateId();
			Gate gate = gatesHasFace.get(gateId);
			
			chart.getSeries(manTime).adjustOrPutValueByCoordinate(gate.getName(),face.getCustomMantime());
			chart.getSeries(manCount).adjustOrPutValueByCoordinate(gate.getName(),face.getCustomCount());
		}
		
		return chart;
	}
	
	
	/**
	 * 人脸性别统计详情
	 * @param faceDatas
	 * @param dataMap
	 * @param reportChart
	 * @return
	 */
	private Chart faceGenderRatioReport(Long[] orgIds,List<GateDayFaceRecognitionSta> faceDatas, Map<String, Object> dataMap, ReportChart reportChart){
		final String male = LocalMessageUtil.getMessage("ParamName.male");
		final String female = LocalMessageUtil.getMessage("ParamName.female");

		Chart chart = new Chart(reportChart.getTitle(), SeriesType.bar);
		Axis<String> xAxis = AxisFactory.createStringAxis();
		Map<Long, Gate> gatesHasFace = getGatesHasFace(orgIds, dataMap);
		for (Gate gate : gatesHasFace.values()) {
			xAxis.addData(gate.getName());
		}
		chart.setXAxis(xAxis);
		
		for (GateDayFaceRecognitionSta face : faceDatas) {
			Long gateId = face.getGateId();
			Gate gate = gatesHasFace.get(gateId);
			chart.getSeries(male).adjustOrPutValueByCoordinate(gate.getName(), face.getMaleCount());
			chart.getSeries(female).adjustOrPutValueByCoordinate(gate.getName(), face.getFemaleCount());
		}
		
		return chart;
	}
	
	/**
	 * 人脸年龄统计详情
	 * @param faceDatas
	 * @param dataMap
	 * @param reportChart
	 * @return
	 */
	private Chart faceAgeRatioReport(Long[] orgIds,List<GateDayFaceRecognitionSta> faceDatas, Map<String, Object> dataMap, ReportChart reportChart){
		Chart chart = new Chart(reportChart.getTitle(), SeriesType.bar){
			@Override
			public Object calcValue(String name, int index, List data_raw) {
				if(data_raw.get(index) == null)
					return 0;
				return super.calcValue(name, index, data_raw);
			}
		};
		
		Axis<String> xAxis = AxisFactory.createStringAxis();
		Map<Long, Gate> gatesHasFace = getGatesHasFace(orgIds, dataMap);
		for (Gate gate : gatesHasFace.values()) {
			xAxis.addData(gate.getName());
		}
		chart.setXAxis(xAxis);

		String[] ages = ageStage.split(",");
		List<Integer> ageStages = new ArrayList<>();
		for (String age : ages) {
			Integer ageInt = Integer.parseInt(age);
			ageStages.add(ageInt);
		}
		String[] ageThresholdName = AgeProcessUtil.calAgeThresholdName(ageStages);
		for (String ageStr : ageThresholdName) {
			chart.createSeries(ageStr);
		}
		
		for (GateDayFaceRecognitionSta face : faceDatas) {
			Long gateId = face.getGateId();
			Gate gate = gatesHasFace.get(gateId);

			String maleStage = face.getMaleStage();
			String femaleStage = face.getFemaleStage();
			if( maleStage==null ||  maleStage.isEmpty() ||  femaleStage==null || femaleStage.isEmpty()){
				continue;
			}
			String[] maleAgestatic = maleStage.split(",", -2);
			String[] femaleAgestatic = femaleStage.split(",", -2);
			if (maleAgestatic.length != femaleAgestatic.length) {
				continue;
			}
			for (int rangeNum = 0; rangeNum < maleAgestatic.length; rangeNum++) {
				int maleNum = Integer.parseInt("" + maleAgestatic[rangeNum].trim());
				int femaleNum = Integer.parseInt("" + femaleAgestatic[rangeNum].trim());
				String ageRange = ageThresholdName[AgeProcessUtil.getIndexByAge(rangeNum,ageStages)];
				chart.getSeries(ageRange).adjustOrPutValueByCoordinate(gate.getName(), NumberUtil.valueAdd(femaleNum,maleNum));
			}

		}
		
		return chart;
	}

	/**
	 * 获取监控点天级人脸分析数据
	 * @param gateIds
	 * @param startDate
	 * @param endDate
	 * @param dataMap
	 * @return
	 */
	private List<GateDayFaceRecognitionSta> getGateDayFaceDatas(Long[] gateIds,Date startDate,Date endDate,Map<String, Object> dataMap){
		List<GateDayFaceRecognitionSta> faceDatas = (List<GateDayFaceRecognitionSta>) dataMap.get(FELID_FACE_DATA);
		if (faceDatas == null){
			faceDatas = new ArrayList<>();
			GateDayFaceRecognitionStaExample gateDayFaceRecognitionStaExample = new GateDayFaceRecognitionStaExample();
			gateDayFaceRecognitionStaExample.createCriteria().andGateIdIn(Arrays.asList(gateIds)).andCounttimeGreaterThanOrEqualTo(startDate)
					.andCounttimeLessThanOrEqualTo(endDate);
			faceDatas = gateDayFaceRecognitionStaService.selectByExample(gateDayFaceRecognitionStaExample);
			dataMap.put(FELID_FACE_DATA,faceDatas);
		}
		return  faceDatas;
	}
	
	/**
	 * 获取有人脸的监控点ID
	 * @param orgIds
	 * @param dataMap
	 * @return
	 */
	private Long[] getGateHasFace(Long[] orgIds,Map<String, Object> dataMap){
		Map<Long,Gate> gateMaps = (Map<Long,Gate>) dataMap.get(FELID_GATE_HAS_FACE);
		
		OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
		if(gateMaps == null){
			if(orgType == OrgType.mall){
				gateMaps = getGatesHasFace(orgIds, dataMap);
			}
		}
		Long[] gateIds = null;
		if(orgType == OrgType.mall){
			gateIds =(Long[]) gateMaps.keySet().toArray();
		}else if(orgType == OrgType.gate){
			gateIds = orgIds;
		}
		return gateIds;
	}

	/**
	 * 获取有人脸设备的监控点对象列表
	 * @param orgIds
	 * @param dataMap
	 * @return
	 */
	private Map<Long,Gate> getGatesHasFace(Long[] orgIds,Map<String, Object> dataMap){
		Map<Long,Gate> gateMap =  (Map<Long, Gate>) dataMap.get(FELID_GATE_HAS_FACE);
		if(gateMap == null){
			gateMap = new HashMap<Long, Gate>();
			OrgType orgType = (OrgType) dataMap.get("ORGTYPE");
			GateExample gateExample = new GateExample();
			if(orgType == OrgType.mall){
				gateExample.createCriteria().andIsHasFaceEqualTo(PARAM_GATE_HAS_FACE).andMallIdIn(Arrays.asList(orgIds));
				
			}else if(orgType == OrgType.gate){
				gateExample.createCriteria().andIsHasFaceEqualTo(PARAM_GATE_HAS_FACE).andIdIn(Arrays.asList(orgIds));
			}
			List<Gate> gates = gateService.selectByExample(gateExample);
			if(gates == null){
				gates = new ArrayList<Gate>();
			}
			for (Gate gate : gates) {
				gateMap.put(gate.getId(), gate);
			}
			dataMap.put(FELID_GATE_HAS_FACE, gateMap);
		}
		return gateMap;
	}

	@Override
	public Map<String, Object> getHead(Long[] orgIds, Date startDate, Date endDate, Map<String, Object> dataMap) {
		return null;
	}

}
