package com.viontech.keliu.util;


import org.springframework.http.MediaType;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;


/**
 * @author 谢明辉
 */

public class HttpUtil {

    /**
     * 将http请求转换为map
     *
     * @param request http请求
     * @return java.util.Map<java.lang.String, java.lang.Object>
     */
    public static Map<String, Object> requestConvert2Map(HttpServletRequest request) {
        Map<String, Object> map = new HashMap<>();
        // request line
        map.put("method", request.getMethod());
        map.put("uri", organizeUri(request));

        // request header
        organizeHeaders(map, request);

        // request body
        if (request.getHeader("Content-Type") != null && request.getHeader("Content-Type").contains(MediaType.APPLICATION_JSON.toString())) {
            map.put("req_body", getRequestBody(request));
        }

        return map;
    }

    /**
     * 将header写入map中
     *
     * @param map     map
     * @param request request
     */
    private static void organizeHeaders(Map<String, Object> map, HttpServletRequest request) {
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = headerNames.nextElement();
            switch (headerName) {
                case "host":
                case "accept":
                case "connection":
                    map.put(headerName, request.getHeader(headerName));
                    break;
                case "accept-encoding":
                    map.put("accept_encoding", request.getHeader(headerName));
                    break;
                case "accept-language":
                    map.put("accept_language", request.getHeader(headerName));
                    break;
                case "content-type":
                    map.put("content_type", request.getHeader(headerName));
                    break;
                case "content-length":
                    map.put("content_length", Integer.parseInt(request.getHeader(headerName)));
                    break;
                case "serial_num":
                    map.put("serial_num", request.getHeader(headerName));
                    break;
                default:
                    map.put(headerName, request.getHeader(headerName));
                    break;
            }
        }
    }

    /**
     * 拼接完整带参数url
     *
     * @param request 请求
     * @return java.lang.String
     */
    private static String organizeUri(HttpServletRequest request) {
        String uri = request.getRequestURI();
        Enumeration<String> parameterNames = request.getParameterNames();
        if (!parameterNames.hasMoreElements()) {
            return uri;
        }
        StringBuilder sb = new StringBuilder(uri).append("?");
        while (parameterNames.hasMoreElements()) {
            String name = parameterNames.nextElement();
            sb.append(name).append("=").append(request.getParameter(name)).append("&");
        }
        sb.deleteCharAt(sb.length() - 1);
        return sb.toString();
    }

    /**
     * 获取requestBody
     *
     * @param request request
     * @return java.lang.String
     */
    private static String getRequestBody(HttpServletRequest request) {
        BufferedReader br = null;
        StringBuilder sb = new StringBuilder();
        try {

            br = request.getReader();
            String s;
            while ((s = br.readLine()) != null) {
                sb.append(s);
            }

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return sb.toString();
    }
}
