package com.viontech.keliu.netty;


import com.viontech.keliu.model.DeviceInfo;
import com.viontech.keliu.model.Message;
import io.netty.channel.Channel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;


/**
 * @author 谢明辉
 */

public class ChannelGroup {

    private static Logger log = LoggerFactory.getLogger(ChannelGroup.class);
    private static Lock LOCK = new ReentrantLock();

    /** 设备序列号和通道id的映射 */
    private static final Map<String, Channel> SERIAL_NUM_CHANNEL_MAP = new LinkedHashMap<>();
    /** 设备序列号和最后心跳的映射 */
    public static final Map<String, Long> HEART_BEAT_MAP = new LinkedHashMap<>();
    /** 消息id集合 */
    public static final LinkedHashMap<Integer, String> ID_DATA_MAP = new LinkedHashMap<>();
    /** 设备序列号和设备信息map */
    public static final LinkedHashMap<String, DeviceInfo> DEVICE_INFO_MAP = new LinkedHashMap<>();

    /**
     * 设备注册
     *
     * @param serialNum 设备序列号
     * @param channel   通道
     */
    public static void registered(String serialNum, Channel channel) {
        try {
            LOCK.lock();
            SERIAL_NUM_CHANNEL_MAP.put(serialNum, channel);
        } finally {
            LOCK.unlock();
        }
        log.info("设备注册成功:{}", serialNum);

    }

    /**
     * @param serialNum 指定设备序列号
     * @param msg       要发送的消息
     */
    public static void sendMessage(String serialNum, Message msg) throws InterruptedException {
        Channel channel = SERIAL_NUM_CHANNEL_MAP.get(serialNum);
        channel.writeAndFlush(msg).sync();
    }


    /**
     * @param channel 解除注册的设备的channel
     */
    public static void unregistered(Channel channel) {
        try {
            LOCK.lock();
            for (Map.Entry<String, Channel> entry : SERIAL_NUM_CHANNEL_MAP.entrySet()) {
                if (entry.getValue().id().toString().equals(channel.id().toString())) {
                    SERIAL_NUM_CHANNEL_MAP.remove(entry.getKey());
                    log.info("设备{}下线", entry.getKey());
                    return;
                }
            }
        } finally {
            LOCK.unlock();
        }

    }

    /**
     * 根据通道获取设备序列号
     *
     * @param channel 通道
     * @return java.lang.String
     */
    public static String getSerialnum(Channel channel) {
        try {
            LOCK.lock();
            for (Map.Entry<String, Channel> entry : SERIAL_NUM_CHANNEL_MAP.entrySet()) {
                if (entry.getValue().id().toString().equals(channel.id().toString())) {
                    return entry.getKey();
                }
            }
        } finally {
            LOCK.unlock();
        }
        return null;
    }

    /**
     * @param serialNum channel对应的设备序列号
     * @return io.netty.channel.Channel
     */
    public static Channel getChannel(String serialNum) {
        return SERIAL_NUM_CHANNEL_MAP.get(serialNum);
    }

    /**
     * 获取设备序列号列表
     *
     * @return java.util.List<java.lang.String>
     */
    public static List<DeviceInfo> getSerialNumList() {
        return SERIAL_NUM_CHANNEL_MAP.keySet().stream()
                .map(DEVICE_INFO_MAP::get).collect(Collectors.toList());
    }

    /**
     * 根据设备序列号移除设备
     *
     * @param serialnum 设备序列号
     */
    public static void unregistered(String serialnum) {
        try {
            LOCK.lock();
            SERIAL_NUM_CHANNEL_MAP.remove(serialnum);
        } finally {
            LOCK.unlock();
        }
    }
}
