package com.viontech.keliu.controller;


import com.viontech.keliu.Contains;
import com.viontech.keliu.enums.CodeEnum;
import com.viontech.keliu.enums.FlagEnums;
import com.viontech.keliu.enums.HttpRespCodeEnum;
import com.viontech.keliu.model.Message;
import com.viontech.keliu.model.MsgID;
import com.viontech.keliu.netty.ChannelGroup;
import com.viontech.keliu.service.ForwardService;
import com.viontech.keliu.util.JsonMessageUtil;
import com.viontech.keliu.util.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;


/**
 * @author 谢明辉
 */
@RestController()
public class ForwardController {
    private static final Lock ATOKEN_LOCK = new ReentrantLock();

    private final static Map<String, Long> ATOKEN_MAP = new ConcurrentHashMap<>();

    private static final Logger log = LoggerFactory.getLogger(ForwardController.class);

    @Value("${auth_url:http://127.0.0.1:9191}")
    private String authUrlPre;
    @Resource
    private ForwardService forwardService;

    @RequestMapping("/**")
    public Object main(HttpServletRequest request, HttpServletResponse response) throws Exception {
        //请求设备信息列表
        if (request.getRequestURI().contains(Contains.URL_CONTAIN_DEVICE_LIST)) {
            return ChannelGroup.getSerialNumList();
        }
        //文件下载
        if (request.getRequestURI().contains(Contains.URL_CONTAIN_DOWNLOAD_FILE)) {
            forwardService.downloadFile(request, response);
            return null;
        }
        // todo 上线时打开  atoken验证
//        String atoken = request.getHeader("atoken");
//        if (!verifyAtoken(atoken)) {
//            return Util.getErrorMessage(HttpRespCodeEnum.INCORRECT_ATOKEN);
//        }

        //判断这个设备的心跳，心跳超时则删除设备
        String serialNum = request.getHeader("serialNum");
        Long heart = ChannelGroup.HEART_BEAT_MAP.get(serialNum);
        if (System.currentTimeMillis() - Optional.ofNullable(heart).orElse(System.currentTimeMillis()) > TimeUnit.MINUTES.toMillis(5)) {
            log.info("无法找到设备序列号:{}", serialNum);
            ChannelGroup.unregistered(serialNum);
        }

        if (ChannelGroup.getChannel(serialNum) == null) {
            return Util.getErrorMessage(HttpRespCodeEnum.DEVICE_NOT_EXIST);
        }

        // ===============================================从这里开始处理指向设备的请求=========================

        //根据请求的路径获取对应flag
        FlagEnums flag = getFlag(request);

        //处理request并得到处理后的request的字符串
        String data = forwardService.request2String(request, flag);

        //向对应设备发送请求,等待返回的消息.请求返回后,得到response的hashMap,截图需要重构response_body
        HashMap responseMap;
        try {
            responseMap = sendMessageAndGetResponse(serialNum, data, flag);
        } catch (RuntimeException e) {
            e.printStackTrace();
            return JsonMessageUtil.getErrorJsonMsg(e.getMessage());
        }

        if (Integer.parseInt(String.valueOf(responseMap.get("status_code"))) != CodeEnum.HTTP_CODE_200.value) {
            return JsonMessageUtil.getErrorJsonMsg((String) responseMap.get("status_msg"));
        }

        return getResponseBody(responseMap, flag, serialNum);
    }

    /**
     * 通过判断请求路径得到对应的flag
     *
     * @param request 请求
     * @return com.viontech.keliu.enums.FlagEnums
     */
    private FlagEnums getFlag(HttpServletRequest request) {
        String uri = request.getRequestURI();
        if (uri.contains(Contains.URL_CONTAIN_UPGRADE_FIRMWARE)) {
            return FlagEnums.UPLOAD_FLAG;
        }
        if (uri.contains(Contains.URL_CONTAIN_UPGRADE_MODEL)) {
            return FlagEnums.UPLOAD_FLAG;
        }
        if (uri.contains(Contains.URL_CONTAIN_SCREENSHOT)) {
            return FlagEnums.SCREENSHOT_FLAG;
        }
        if (uri.contains(Contains.URL_CONTAIN_DOWNLOAD_LOG)) {
            return FlagEnums.DOWNLOAD_FLAG;
        }
        if (uri.contains(Contains.URL_CONTAIN_SETTINGS_BACKUP)) {
            return FlagEnums.DOWNLOAD_FLAG;
        }

        return FlagEnums.GENERAL_FLAG;
    }


    /**
     * 向设备适配端发送从浏览器接收的请求，并等待返回结果
     *
     * @param serialNum   设备序列号
     * @param requestData String类型需要发送的数据
     * @param flag        flag
     * @return HashMap
     */
    private HashMap sendMessageAndGetResponse(String serialNum, String requestData, FlagEnums flag) throws InterruptedException, IOException {
        // 准备发送tcp请求
        int msgId = MsgID.getId();
        ChannelGroup.ID_DATA_MAP.put(msgId, null);

        int commandValue;
        if (flag == FlagEnums.GENERAL_FLAG) {
            commandValue = CodeEnum.COMMAND_CONTROL.value;
        } else if (flag == FlagEnums.UPLOAD_FLAG) {
            commandValue = CodeEnum.COMMAND_UPLOAD_FILE.value;
        } else {
            commandValue = CodeEnum.COMMAND_GENERAL_COMMAND.value;
        }

        // 构建message
        Message message;
        if (flag == FlagEnums.UPLOAD_FLAG) {
            message = new Message(CodeEnum.DEFAULT_VERSION.value, commandValue, msgId, Base64.getDecoder().decode(requestData));
        } else {
            log.info("发送请求:{}", requestData);
            message = new Message(CodeEnum.DEFAULT_VERSION.value, commandValue, msgId, requestData.getBytes(Charset.forName("utf-8")));
        }

        // 发送消息
        ChannelGroup.sendMessage(serialNum, message);

        // 等待返回的数据
        long begin = System.currentTimeMillis();
        while (ChannelGroup.ID_DATA_MAP.get(msgId) == null) {
            Thread.sleep(200);
            if ((System.currentTimeMillis() - begin) > TimeUnit.MINUTES.toMillis(30)) {
                throw new RuntimeException("请求超时");
            }
        }
        // 移除
        String responseData = ChannelGroup.ID_DATA_MAP.get(msgId);
        ChannelGroup.ID_DATA_MAP.remove(msgId);
        // 返回
        HashMap allResposne = Util.json2Map(responseData);
        HashMap deviceResponse = (HashMap) allResposne.get("response");
        if (flag.equals(FlagEnums.DOWNLOAD_FLAG) || flag.equals(FlagEnums.SCREENSHOT_FLAG)) {
            deviceResponse.put("file_name", allResposne.get("file_name"));
        }
        return deviceResponse;
    }

    /**
     * 向浏览器返回设备返回的信息
     *
     * @param response  返回的信息
     * @param flag      指令flag
     * @param serialnum 设备序列号
     * @return java.lang.Object
     */
    private Object getResponseBody(HashMap response, FlagEnums flag, String serialnum) throws IOException {
        if (flag == FlagEnums.SCREENSHOT_FLAG) {
            return forwardService.saveScreenShotFromResponse(response, serialnum);
        }
        if (flag == FlagEnums.DOWNLOAD_FLAG) {
            return forwardService.saveOtherFileFromResponse(response, serialnum);
        }
        return response.get("response_body");
    }


    /**
     * 验证atoken，内存缓存10分钟，超时需要向auth服务验证
     *
     * @param atoken atoken
     * @return 通过返回true
     */
    private boolean verifyAtoken(String atoken) {
        if (atoken == null) {
            return false;
        }
        try {
            ATOKEN_LOCK.lock();

            Long old = ATOKEN_MAP.get(atoken);
            if (old != null) {
                if (System.currentTimeMillis() - old < TimeUnit.MINUTES.toMillis(30)) {
                    return true;
                } else {
                    ATOKEN_MAP.remove(atoken);
                }
            }
        } finally {
            ATOKEN_LOCK.unlock();
        }

        String url = authUrlPre + "/api/v1/auth/users/atoken/" + atoken;
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.add("effect", "operation");
        httpHeaders.add("atoken", atoken);
        HttpEntity<String> entity = new HttpEntity<>("", httpHeaders);
        try {
            ResponseEntity<HashMap> response = restTemplate.exchange(url, HttpMethod.GET, entity, HashMap.class);
            if (response.getBody() == null || response.getBody().get("username") == null) {
                return false;
            }
        } catch (Exception e) {
            log.error("atoken校验时出错");
            return false;
        }

        ATOKEN_MAP.put(atoken, System.currentTimeMillis());
        return true;
    }

}

