package com.viontech.keliu.util;



import com.viontech.keliu.util.trove.list.array.TDoubleArrayList;
import com.viontech.keliu.util.trove.list.array.TIntArrayList;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.regex.Pattern;

public class NumberUtil {

    /**
     * 判断当前值是否为整数
     *
     * @param value
     * @return
     */
    public static boolean isInteger(Object value) {
        if (value == null || value.toString().isEmpty()) {
            return false;
        }
        String mstr = value.toString();
        Pattern pattern = Pattern.compile("^-?\\d+{1}");
        return pattern.matcher(mstr).matches();
    }

    /**
     * 判断当前值是否为数字(包括小数)
     *
     * @param value
     * @return
     */
    public static boolean isDigit(Object value) {
        if (value == null || value.toString().isEmpty()) {
            return false;
        }
        String mstr = value.toString();
        Pattern pattern = Pattern.compile("^-?[0-9]*.?[0-9]*{1}");
        return pattern.matcher(mstr).matches();
    }

    /**
     * 将数字格式化输出
     *
     * @param value     需要格式化的值
     * @param precision 精度(小数点后的位数)
     * @return
     */
    public static String format(Object value, Integer precision) {
        Double number = 0.0;
        if (NumberUtil.isDigit(value)) {
            number = new Double(value.toString());
        }
        precision = (precision == null || precision < 0) ? 2 : precision;
        BigDecimal bigDecimal = new BigDecimal(number);
        return bigDecimal.setScale(precision, BigDecimal.ROUND_HALF_UP)
                .toString();
    }

    /**
     * 将数字格式化输出
     *
     * @param value
     * @return
     */
    public static String format(Object value) {
        return NumberUtil.format(value, 2);
    }

    /**
     * 将值转成Integer型，如果不是整数，则返回0
     *
     * @param value
     * @param replace 如果为0或者null，替换值
     * @return
     */
    public static Integer parseInteger(Object value, Integer replace) {
        if (!NumberUtil.isInteger(value)) {
            return replace;
        }
        return new Integer(value.toString());
    }

    /**
     * 将值转成Integer型，如果不是整数，则返回0
     *
     * @param value
     * @return
     */
    public static Integer parseInteger(Object value) {
        return NumberUtil.parseInteger(value, 0);
    }

    /**
     * 将值转成Long型
     *
     * @param value
     * @param replace 如果为0或者null，替换值
     * @return
     */
    public static Long parseLong(Object value, Long replace) {
        if (!NumberUtil.isInteger(value)) {
            return replace;
        }
        return new Long(value.toString());
    }

    /**
     * 将值转成Long型，如果不是整数，则返回0
     *
     * @param value
     * @return
     */
    public static Long parseLong(Object value) {
        return NumberUtil.parseLong(value, 0L);
    }

    /**
     * 将值转成Double型
     *
     * @param value
     * @param replace replace 如果为0或者null，替换值
     * @return
     */
    public static Double parseDouble(Object value, Double replace) {
        if (!NumberUtil.isDigit(value)) {
            return replace;
        }
        return new Double(value.toString());
    }

    /**
     * 将值转成Double型，如果不是整数，则返回0
     *
     * @param value
     * @return
     */
    public static Double parseDouble(Object value) {
        return NumberUtil.parseDouble(value, 0.0);
    }

    /**
     * 将char型数据转成字节数组
     *
     * @param value
     * @return
     */
    public static byte[] toBytes(char value) {
        byte[] bt = new byte[2];
        for (int i = 0; i < bt.length; i++) {
            bt[i] = (byte) (value >>> (i * 8));
        }
        return bt;
    }

    /**
     * 将short型数据转成字节数组
     *
     * @param value
     * @return
     */
    public static byte[] toBytes(short value) {
        byte[] bt = new byte[2];
        for (int i = 0; i < bt.length; i++) {
            bt[i] = (byte) (value >>> (i * 8));
        }
        return bt;
    }

    /**
     * 将int型数据转成字节数组
     *
     * @param value
     * @return
     */
    public static byte[] toBytes(int value) {
        byte[] bt = new byte[4];
        for (int i = 0; i < bt.length; i++) {
            bt[i] = (byte) (value >>> (i * 8));
        }
        return bt;
    }

    /**
     * 将long型数据转成字节数组
     *
     * @param value
     * @return
     */
    public static byte[] toBytes(long value) {
        byte[] bt = new byte[8];
        for (int i = 0; i < bt.length; i++) {
            bt[i] = (byte) (value >>> (i * 8));
        }
        return bt;
    }

    /**
     * 将short型数据插入到指定索引的字节数组中
     *
     * @param index  索引
     * @param values 字节数组
     * @param value  需要插入的值
     */
    public static void insert(int index, byte[] values, short value) {
        byte[] bt = NumberUtil.toBytes(value);
        System.arraycopy(bt, 0, values, index, 2);
    }

    /**
     * 将int型数据插入到指定索引的字节数组中
     *
     * @param index  索引
     * @param values 字节数组
     * @param value  需要插入的值
     */
    public static void insert(int index, byte[] values, int value) {
        byte[] bt = NumberUtil.toBytes(value);
        System.arraycopy(bt, 0, values, index, 4);
    }

    /**
     * 将long型数据插入到指定索引的字节数组中
     *
     * @param index  索引
     * @param values 字节数组
     * @param value  需要插入的值
     */
    public static void insert(int index, byte[] values, long value) {
        byte[] bt = NumberUtil.toBytes(value);
        System.arraycopy(bt, 0, values, index, 8);
    }

    /**
     * 将字节转换成整型
     *
     * @param value 字节类型值
     * @return
     */
    public static int byteToInt(byte value) {
        if (value < 0) {
            return value + 256;
        }
        return value;
    }

    /**
     * 向上舍入
     *
     * @param f值
     * @param digit保留位数
     * @return
     */
    public static float formatRoundUp(float number, int digit) {
        BigDecimal b = new BigDecimal(number);
        /**四舍五入保留两位小数*/
        return b.setScale(digit, BigDecimal.ROUND_UP).floatValue();
    }

    /**
     * 向上舍入
     *
     * @param f值
     * @param digit保留位数
     * @return
     */
    public static double formatRoundUp(double number, int digit) {
        BigDecimal b = new BigDecimal(number);
        /**四舍五入保留两位小数*/
        return b.setScale(digit, BigDecimal.ROUND_UP).doubleValue();
    }


    /**
     * 四舍五入
     *
     * @param number 值
     * @param digit  保留位数
     * @return
     */
    public static float formatRoundHalfUp(float number, int digit) {
        BigDecimal b = new BigDecimal(number);
        /**四舍五入保留两位小数*/
        return b.setScale(digit, BigDecimal.ROUND_HALF_UP).floatValue();
    }

    /**
     * 四舍五入
     *
     * @param number 值
     * @param digit  保留位数
     * @return
     */
    public static Double formatRoundHalfUp(double number, int digit) {
        BigDecimal b = new BigDecimal(number);
        /**四舍五入保留两位小数*/
        return b.setScale(digit, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 向上舍入
     *
     * @param number 需要四舍五入的数值
     * @param digit  保留位数
     * @return 四舍五入后的结果
     */
    public static BigDecimal formatRoundUp(BigDecimal number, int digit) {
        /**四舍五入保留两位小数*/
        return number.setScale(digit, BigDecimal.ROUND_UP);
    }

    /**
     * 四舍五入
     *
     * @param number 需要四舍五入的数值
     * @param digit  保留位数
     * @return 四舍五入后的结果
     */
    public static BigDecimal formatRoundHalfUp(BigDecimal number, int digit) {
        /**四舍五入保留两位小数*/
        return number.setScale(digit, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 计算均值
     *
     * @param values
     * @return
     */
    public static Integer getAvg(TIntArrayList values) {
        int sum = values.sum();
        int count = values.size();
        Double result = divide(sum, count, 0);
        if (result == null) {
            return null;
        }
        return result.intValue();
    }

    /**
     * 计算均值
     *
     * @param values
     * @return
     */
    public static double getAvg(TDoubleArrayList values, int digit) {
        double sum = values.sum();
        int count = values.size();
        return divide(sum, count, digit);
    }

    /**
     * 除法计算
     *
     * @param dividend 除数
     * @param divisor  被除数
     * @param digit    保留位数
     * @return
     */
    public static Double divide(Number dividend, Number divisor, int digit) {
        if (dividend == null || divisor == null) return null;
        BigDecimal dividendBD = new BigDecimal(dividend.toString());
        BigDecimal divisorBD = new BigDecimal(divisor.toString());
        try {
            return dividendBD.divide(divisorBD, digit, BigDecimal.ROUND_HALF_UP).doubleValue();
        } catch (Exception e) {
            //e.printStackTrace();
        }
        return null;
    }

    /**
     * 减法计算
     *
     * @param minuend    减数
     * @param subtrahend 被减数
     * @return
     */
    public static Double subtract(Number minuend, Number subtrahend) {
        if (minuend == null || subtrahend == null) return null;
        BigDecimal minuendBD = new BigDecimal(minuend.toString());
        BigDecimal subtrahendBD = new BigDecimal(subtrahend.toString());
        try {
            return minuendBD.subtract(subtrahendBD).doubleValue();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0.0D;
    }

    /**
     * 计算增长率（ newData-oldData）/oldData*100
     *
     * @param newData
     * @param oldData
     * @return
     */
    public static String growthRate(Number newData, Number oldData) {
        if (newData == null || oldData == null) {
            return null;
        }
        Double rd = divide(NumberUtil.subtract(newData, oldData), oldData, 6);
        if (rd == null) {
            return null;
        }
        String result = formatRoundHalfUp(rd * 100, 2) + "%";
        return result;
    }

    /**
     * 计算百分比（ newData-oldData）/oldData*100
     *
     * @param newData
     * @param oldData
     * @return
     */
    public static Double percentage(Number dividend, Number divisor, int digit) {
        if (dividend == null || divisor == null) {
            return null;
        }
        Double result = divide(dividend, divisor, digit + 2);
        if (result == null) {
            return null;
        }
        result = formatRoundHalfUp(result * 100, digit);
        return result;
    }

    public static BigDecimal conver2BigDecimal(Number number) {
        if (number == null) return null;
        return new BigDecimal(number.toString());
    }

    public static Object valueAdd(Object t1, Object t2) {
        if (t1 == null) return t2;
        if (t2 == null) return t1;
        BigDecimal sum = new BigDecimal(t1.toString()).add(new BigDecimal(t2.toString()));
        if (t1 instanceof Double) {
            return (Object) new Double(sum.doubleValue());
        }
        if (t1 instanceof Integer) {
            return new Integer(sum.intValue());
        }
        if (t1 instanceof Float) {
            return new Float(sum.floatValue());
        }
        if (t1 instanceof Long) {
            return new Long(sum.longValue());
        }
        return new Double(sum.doubleValue());
    }


    /**
     * 数值是否是0
     *
     * @param number number
     * @return true or false
     */
    public static boolean isZero(Number number) {
        if (number == null) {
            return false;
        }
        if (number instanceof BigDecimal) {
            return ((BigDecimal) number).compareTo(BigDecimal.ZERO) == 0;
        } else if (number instanceof BigInteger) {
            return ((BigInteger) number).compareTo(BigInteger.ZERO) == 0;
        } else if (number instanceof Float) {
            return number.floatValue() == 0.0f;
        } else {
            return number.doubleValue() == 0.0;
        }
    }

}
