package com.viontech.keliu.base;

import cn.monitor4all.logRecord.annotation.OperationLog;
import cn.monitor4all.logRecord.context.LogRecordContext;
import com.github.pagehelper.PageInfo;
import com.viontech.keliu.util.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;

import static com.viontech.keliu.i18n.util.LocalMessageUtil.getMessage;
import static com.viontech.keliu.util.JsonMessageUtil.getErrorJsonMsg;
import static com.viontech.keliu.util.JsonMessageUtil.getSuccessJsonMsg;

/**
 * Controller基类<br>
 * 所有Controller都需要继承自该类
 *
 * @author suman 2016年8月15日 下午1:59:19
 * @param <T>
 */

public abstract class BaseController<O extends  BaseModel,T extends VoInterface<O>> {


	/**
	 * slf4j 日志对象 用来记录log
	 */
	protected final Logger logger = LoggerFactory.getLogger(getClass());
	/**
	 * 通用message
	 */
	public static final String MESSAGE_ADD_SUCCESS = "addSuccess";
	public static final String MESSAGE_ADD_ERROR = "addError";
	public static final String MESSAGE_UPDATE_SUCCESS = "updateSuccess";
	public static final String MESSAGE_UPDATE_ERROR = "updateError";
	public static final String MESSAGE_DELETE_SUCCESS= "delSuccess";
	public static final String MESSAGE_DELETE_ERROR = "delError";
	public static final String MESSAGE_COUNT_SUCCESS = "countSuccess";
	public static final String MESSAGE_COUNT_ERROR = "countError";
	public static final String MESSAGE_SELECT_SUCCESS = "selSuccess";
	public static final String MESSAGE_SELECT_ERROR = "selError";
	public static final String MESSAGE_LIST_SUCCESS = "listSuccess";
	public static final String MESSAGE_LIST_ERROR = "listError";
	public static final String MESSAGE_PAGE_SUCCESS	= "pageSuccess";
	public static final String MESSAGE_PAGE_ERROR = "pageError";
	public static final String MESSAGE_WEBROOT_NOTFOUND_ERROR = "webRootNotFoundError";
	public static final String MESSAGE_FILE_UPLOAD_SUCCESS = "fileUploadSuccess";
	public static final String MESSAGE_FILE_UPLOAD_ERROR = "fileUploadError";
	public static final String MESSAGE_FILE_DOWNLOAD_ERROR = "fileDownloadError";
	public static final String MESSAGE_FILE_NOTFOUND_ERROR = "fileNotFoundError";
	public static final String MESSAGE_ID_NOT_EMPTY = "idNotEmpty";
	public static final String MESSAGE_MODE_NOT_EXISTENCE = "modleNotExistence";

	public static final int EXAMPLE_TYPE_SIMPLE = 0;
	public static final int EXAMPLE_TYPE_NORMAL = 1;
	public static final int EXAMPLE_TYPE_COUNT = 3;
	public static final int EXAMPLE_TYPE_PAGE = 4;

	/*@InitBinder
	public void initBinder(WebDataBinder binder) {
		binder.registerCustomEditor(Date.class, new CustomDateEditor(true,
				"yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd", "yyyy-MM", "HH:mm:ss"));
	}*/

	/**
	 * 通用添加方法
	 *
	 * @param t
	 *            需要插入数据库的对象
	 * @return
	 */
	@RequestMapping(value="",method=RequestMethod.POST)
	@ResponseBody
	@OperationLog(bizId = "#t.getModel().getId()", bizType = "#className", msg = "#className + '新增'")
	public Object add(@RequestBody  T t) {
		getService().insertSelective(t.getModel());
		LogRecordContext.putVariable("className", t.getModel().getClass().getSimpleName());
		return getSuccessJsonMsg(getMessage(MESSAGE_ADD_SUCCESS, getClass()), t);
	}

	/**
	 * 通用数量方法
	 *
	 * @param t 需要插入数据库的对象的条件
	 * @return
	 */
	@RequestMapping(value="/count",method=RequestMethod.GET)
	@ResponseBody
	public Object count(T t) {
		int result = getService().countByExample(getExample(t,EXAMPLE_TYPE_COUNT));
		return getSuccessJsonMsg(getMessage(MESSAGE_COUNT_SUCCESS, getClass()), result);
	}

	/**
	 * 通用更新方法
	 *
	 * @param t 需要更新的对象及内容
	 * @return
	 */

    @RequestMapping(value = "/{id}", method = RequestMethod.POST)
    @ResponseBody
    @OperationLog(bizId = "#id", bizType = "#className", msg = "#className + '信息修改'", extra = "#_DIFF(#oldObject, #newObject)")
    public Object update(@PathVariable(value = "id") Long id, @RequestBody T t) {
        t.getModel().setId(id);
        getService().updateByPrimaryKeySelective(t.getModel());

        BaseModel record = getService().selectByPrimaryKey(id);
        // 手动传递日志上下文
        LogRecordContext.putVariable("oldObject", record);
        LogRecordContext.putVariable("newObject", t.getModel());
        // 获取泛型的 Class 对象

		LogRecordContext.putVariable("className", t.getModel().getClass().getSimpleName());
        return getSuccessJsonMsg(getMessage(MESSAGE_UPDATE_SUCCESS, getClass()), t);
    }

	/**
	 * 通用删除方法
	 *
	 * @param id
	 *            需要删除的数据id
	 * @return
	 */
	@RequestMapping(value = "/{id}",method=RequestMethod.DELETE)
	@ResponseBody
	@OperationLog(bizId = "#id", bizType = "#className", msg = "'删除' + #className")
	public Object del(@PathVariable(value = "id") Long id) {
		if(id == null)
			return getErrorJsonMsg(getMessage(MESSAGE_ID_NOT_EMPTY,getClass()));
		int result = getService().deleteByPrimaryKey(id);
		if(result == 0)
			return getErrorJsonMsg(getMessage(MESSAGE_MODE_NOT_EXISTENCE, getClass()));
		String requestURI = LogRecordContext.getVariable("requestURI").toString();
		String[] split = requestURI.split("/");
		LogRecordContext.putVariable("className", split[split.length - 2]);
		return getSuccessJsonMsg(getMessage(MESSAGE_DELETE_SUCCESS, getClass()), null);
	}

	/**
	 * 通用列表查询方法，将数据从数据库中全部查出
	 * saveFile
	 * @return
	 */
	@RequestMapping(value="/{id}",method=RequestMethod.GET)
	@ResponseBody
	public Object selOne(@PathVariable(value = "id")Long id) {
		BaseModel t = getService().selectByPrimaryKey(id);
		return getSuccessJsonMsg(getMessage(MESSAGE_LIST_SUCCESS, getClass()), t);


	}

	/**
	 * 通用分页查询方法，从数据库中查出一页数据
	 *
	 * @return
	 */
	@RequestMapping(value="",method=RequestMethod.GET)
	@ResponseBody
	public Object page(T t, @RequestParam(value = "page",defaultValue = "-1") int page,@RequestParam(value = "pageSize",defaultValue="100") int pageSize,String sortName,String sortOrder) {
		BaseExample baseExample = getExample(t,EXAMPLE_TYPE_PAGE);
		/*if(isNotNull(sortOrder)&&isNotNull(sortName)){
			baseExample.setOrderByClause(baseExample.getTableAlias()+"."+sortName+" "+sortOrder);
		}else if (isNotNull(sortName) && !isNotNull(sortOrder)){
			baseExample.setOrderByClause(sortName);
		}*/
		if(page <= 0){
			List result = getService().selectByExample(baseExample);
			return getSuccessJsonMsg(getMessage(MESSAGE_SELECT_SUCCESS,getClass()),result);
		}else{
			PageInfo pageInfo = getService().pagedQuery(baseExample, page, pageSize);
			return getSuccessJsonMsg(getMessage(MESSAGE_PAGE_SUCCESS, getClass()),pageInfo);
		}
	}

	/**
	 * 根据参数获取图片，对前后获取图片的不同参数方式进行兼容
	 */
	@RequestMapping(value="/img/{location}/{picture:.+}")
	public void getIMG(@PathVariable("picture")String picture, @PathVariable("location") String location , HttpServletResponse response) {
		if(picture.contains("svg")){
			response.setContentType("image/svg+xml;charset=UTF-8");
			response.addHeader("Accept-Ranges","bytes");
		}else{
			response.setContentType("image/jpeg");
		}

		FileInputStream fis = null;
		OutputStream out = null;
		try {
			String basePath = FileUtil.getBasePath();
			File file = new File(basePath+"/"+location , picture);
			if(!file.exists()){
				System.out.println(file.getAbsolutePath());
				return;
			}
			out = response.getOutputStream();
			fis = new FileInputStream(file);
			byte[] b = new byte[fis.available()];
			response.setContentLength(b.length);
			fis.read(b);
			out.write(b);
			out.flush();
		} catch (IOException e) {
			e.printStackTrace();
		}finally{
			try {
				if(fis != null)
					fis.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
			try {
				if(out !=null){
					out.close();
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * 获取到执行各项操作需要的service
	 *
	 * @return
	 */
	protected abstract BaseService<O> getService();

	/**
	 * 得到执行查询操作需要的查询条件
	 *
	 * @param t
	 *            查询条件存储对象
	 * @return 查询条件对象
	 */
	protected abstract BaseExample getExample(T t,int type);


	public static boolean isNotNull(Object object) {
		if (object == null) {
			return false;
		}
		if (object.toString().trim().isEmpty()) {
			return false;
		}
		return true;
	}


}
