package com.viontech.keliu.oss;

import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.common.utils.IOUtils;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * .
 *
 * @author 谢明辉
 * @date 2020/12/23
 */

public class S3ClientHelper implements ClientHelper {
    private final AmazonS3 amazonS3;
    private final String bucket;
    Logger log = LoggerFactory.getLogger(S3ClientHelper.class);

    public S3ClientHelper(String accessKey, String secretKey, String endPoint, String bucket) {
        this.bucket = bucket;
        AWSCredentials credentials = new BasicAWSCredentials(accessKey, secretKey);
        ClientConfiguration config = new ClientConfiguration();
        config.setProtocol(Protocol.HTTP);
        config.setSignerOverride("S3SignerType");
        config.setSocketTimeout(180000);
        config.setMaxConnections(50);
        config.setConnectionTimeout(10000);
        amazonS3 = AmazonS3ClientBuilder.standard().withCredentials(new AWSStaticCredentialsProvider(credentials)).withClientConfiguration(config).withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(endPoint, Regions.US_EAST_1.getName())).enablePathStyleAccess().build();
    }

    @Override
    public String storeFile(String remoteFile, byte[] content) {
        ByteArrayInputStream inputStream = new ByteArrayInputStream(content);
        String eTag = amazonS3.putObject(bucket, remoteFile, inputStream, null).getETag();
        addAccessRole(bucket, remoteFile);
        return eTag;
    }

    @Override
    public String storeFile(String remoteFile, InputStream stream) {
        String eTag = amazonS3.putObject(bucket, remoteFile, stream, null).getETag();
        addAccessRole(bucket, remoteFile);
        return eTag;
    }

    @Override
    public boolean isExist(String key) {
        return amazonS3.doesObjectExist(bucket, key);
    }

    @Override
    public String getItem(String key) {
        StringBuilder sb = new StringBuilder();
        String s;
        try (BufferedReader br = new BufferedReader(new InputStreamReader(amazonS3.getObject(bucket, key).getObjectContent()))) {
            while ((s = br.readLine()) != null) {
                sb.append(s);
            }
        } catch (IOException e) {
            log.error("", e);
        }
        return sb.toString();
    }

    @Override
    public byte[] getByteArray(String key) {
        try {
            return IOUtils.readStreamAsByteArray(amazonS3.getObject(bucket, key).getObjectContent());
        } catch (IOException e) {
            log.error("oss getByteArray error", e);
            return new byte[0];
        }
    }

    @Override
    public BufferedImage getBufferedImage(String key) {
        try {
            return ImageIO.read(amazonS3.getObject(bucket, key).getObjectContent());
        } catch (IOException e) {
            log.error("oss getBufferedImage error", e);
        }
        return null;
    }

    @Override
    public InputStream getObjectContent(String key) {
        return amazonS3.getObject(bucket, key).getObjectContent();
    }

    @Override
    public byte[] getByteArrayImage(String key) {
        try {
            return IOUtils.readStreamAsByteArray(amazonS3.getObject(bucket, key).getObjectContent());
        } catch (IOException e) {
            log.error("oss getByteArrayImage error", e);
        }
        return null;
    }

    @Override
    public boolean deleteItem(String key) {
        try {
            amazonS3.deleteObject(bucket, key);
            return true;
        } catch (OSSException | ClientException e) {
            log.error("", e);
        }
        return false;
    }

    public void addAccessRole(String bucketName, String key) {
        AccessControlList objectAcl = amazonS3.getObjectAcl(bucketName, key, null);
        Owner owner = objectAcl.getOwner();
        List<Grant> grantList = objectAcl.getGrantsAsList();
        Grant grant = null;
        for (Grant grant1 : grantList) {
            if (owner.getId().equals(grant1.getGrantee().getIdentifier())) {
                grant = grant1;
                break;
            }
        }
        List<Grant> allGrants = new ArrayList<>();
        allGrants.add(grant);
        allGrants.add(new Grant(GroupGrantee.AllUsers, Permission.FullControl));
        AccessControlList accessControlList = new AccessControlList();
        accessControlList.setOwner(owner);
        accessControlList.grantAllPermissions(allGrants.toArray(new Grant[allGrants.size()]));
        amazonS3.setObjectAcl(bucketName, key, null, accessControlList);
    }

}
