package com.viontech.keliu.runner;

import com.viontech.keliu.dao.MallDao;
import com.viontech.keliu.model.ConfigParam;
import com.viontech.keliu.model.Mall;
import com.viontech.keliu.service.ConfigParamService;
import com.viontech.keliu.util.ConfigUtil;
import com.viontech.keliu.websocket.AlgApiClient;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * @Author: 王洪波
 * @Date: 2019/1/11.
 * @Description:历史过期特征库清理任务
 */
@Component
public class FeaturePoolClearJob {

    private Logger logger = LoggerFactory.getLogger(FeaturePoolClearJob.class);

    @Value("${ws.url:}")
    private String url;
    @Value("${spring.application.name:}")
    private String applicationName;
    /** 删除成功返回值 */
    private static final int DELETE_SUCCESS_CODE = 1;
    @Resource
    private ConfigParamService configParamService;
    @Resource
    private MallDao mallDao;
    @Autowired(required = false)
    private AlgApiClient algApiClientComparison;



    @Scheduled(cron = "01 0 06 * * ?")
    public void execute() {
        try{
            if(url.isEmpty()) {
                return;
            }
            //封装配置细节
            Map<Long, ConfigParam> configParamMap = configParamService.selectAll();
            //封装mall配置关系
            Map<String,Long> mallMatchDayMap = new HashMap<>();
            List<Mall> malls = mallDao.selectAll();
            for (Mall mall : malls) {
                String unid = mall.getUnid();
                Long mallId = mall.getId();
                ConfigParam configParam= ConfigUtil.getConfigParam(mallId,configParamMap);
                String newOrRegular = configParam.getNewOrRegular();
                mallMatchDayMap.put(unid,Long.parseLong(newOrRegular));
            }
            logger.info("历史特征库清理任务开始");
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            // 得到今天的日期
            Calendar calendar = Calendar.getInstance();
            calendar.set(Calendar.HOUR_OF_DAY, 0);
            calendar.set(Calendar.SECOND, 0);
            calendar.set(Calendar.MINUTE, 0);
            Date now = calendar.getTime();
            // 请求所有的人员特征池并在response后进行删除
            CompletableFuture<JSONObject> responseFuture = algApiClientComparison.queryPersonPool(1, "customPool,1970-01-01,1", new HashMap<>());
            JSONObject response = responseFuture.get(120, TimeUnit.SECONDS);

            if(response.isNull("poolIds")){
                return;
            }
            JSONArray poolIds = response.getJSONArray("poolIds");
            // 遍历特征池数组
            for (int i = 0; i < poolIds.length(); i++) {
                try {
                    JSONObject obj = poolIds.getJSONObject(i);
                    // 获取pool特征池ID
                    String poolId = obj.getString("poolId");
                    // 判断是否为顾客特征池和是否为自家的特征池
                    if (!poolId.contains("customPool") || !poolId.contains(applicationName)) {
                        continue;
                    }
                    // 根据“_”分割特征池ID，取出日期字符串
                    String[] ss = poolId.split("_");
                    String dateString;
                    if (ss.length > 1) {
                        dateString = ss[ss.length - 2];
                    } else {
                        continue;
                    }
                    // 日期字符串不能为空，并且需要yyyy-MM-dd格式
                    if (dateString ==null || "".equals(dateString) || !dateString.matches("\\d{4}-\\d{2}-\\d{2}")) {
                        continue;
                    }
                    // 将日期字符串转化为日期格式
                    Date date = sdf.parse(dateString);
                    // 根据封装的新老顾客时间规则信息匹配删除
                    String unid = ss[ss.length - 1];
                    Long matchDay = mallMatchDayMap.get(unid);
                    if(matchDay == null || matchDay == 0){//如果获取不到天数设置信息，就默认清除30天以前的顾客特征库。
                        matchDay = 30L;
                    }
                    if ((now.getTime() - date.getTime()) < TimeUnit.DAYS.toMillis(matchDay+3)) {
                        continue;
                    }
                    logger.info("开始删除特征池{}", poolId);
                    System.out.println("开始删除特征库:"+poolId+"匹配天数为:"+matchDay);

                    responseFuture = algApiClientComparison.delPersonPool(1, poolId, new ArrayList<>(), new HashMap<>());
                    response = responseFuture.get(120,TimeUnit.SECONDS);
                    if (response.getInt("success") == DELETE_SUCCESS_CODE) {
                        System.out.println("特征库清除任务-清除特征库:"+poolId);
                        logger.info("删除{}特征池成功", poolId);
                    } else {
                        logger.info("删除{}特征池失败", poolId);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }catch(Exception e){
            e.printStackTrace();
        }
    }

}
