package com.viontech.keliu.batch.item.writer;

import com.viontech.keliu.model.Channel;
import com.viontech.keliu.model.FaceRecognition;
import com.viontech.keliu.model.Gate;
import com.viontech.keliu.model.MallDayHistoryArriveCount;
import com.viontech.keliu.service.OrgCacheService;
import com.viontech.keliu.service.RedissonService;
import com.viontech.keliu.util.DateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ItemStreamException;
import org.springframework.batch.item.ItemStreamWriter;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * .
 *
 * @author 谢明辉
 */
@Component("mallDayHistoryArriveCountWriter")
public class MallDayHistoryArriveCountWriter implements ItemStreamWriter<FaceRecognition> {

    private static final Logger log = LoggerFactory.getLogger(MallDayHistoryArriveCountWriter.class);

    private static final String LOCK_PERSON_UNID = "LOCK_Person_Unid";
    private static final String LOCK_HISTORY_ARRIVE = "LOCK_History_Arrive";
    @Resource
    private RedissonService redissonService;

    @Resource
    private RedisTemplate<String, String> redisTemplate;
    @Resource
    private OrgCacheService orgCacheService;

    @Resource
    private JdbcTemplate jdbcTemplate;

    private final String SQL_INSERT_HISTORY_ARRIVE = "insert into d_mall_day_history_arrive_count(mall_id,account_id,countdate,once,twice,third,fourth,fifth,more) values (?,?,?,?,?,?,?,?,?)";
    private final String SQL_UPDATE_HISTORY_ARRIVE = "update d_mall_day_history_arrive_count set once=once+?,twice=twice+?,third=third+?,fourth=fourth+?,fifth=fifth+?,more=more+? where countdate=? and mall_id=?";
    private final String SQL_SELECT_HISTORY_ARRIVE = "select * from d_mall_day_history_arrive_count where countdate=? and mall_id=?";
    private ThreadLocal<ExecutionContext> executionContextThreadLocal = new ThreadLocal<>();

    @Override
    public void open(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
    }

    @Override
    public void update(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
    }

    @Override
    public void close() throws ItemStreamException {
        executionContextThreadLocal.remove();
    }

    @Override
    public void write(List<? extends FaceRecognition> items) {
        log.info("开始执行历史到店次数统计");
        Map<String, Gate> gateMap = orgCacheService.getGateMap();
        Map channelMap = orgCacheService.getChannelMap();

        Map<Long, MallDayHistoryArriveCount> map = new HashMap<>();
        for (FaceRecognition item : items) {

            if (item.getPersonType() != 0) {
                continue;
            }
            String channelSerialnum = item.getChannelSerialnum();
            Channel channel = (Channel) channelMap.get(channelSerialnum);
            Long gateId = channel.getGateId();
            Gate gate = gateMap.get(String.valueOf(gateId));
            if (gate == null) {
                log.info("监控点不存在，跳过处理:{}", gateId);
                continue;
            }
            ExecutionContext executionContext = executionContextThreadLocal.get();
            Set<Long> firstTimePerson  = (Set<Long>) executionContext.get("firstTimePerson");
            if(firstTimePerson == null || !firstTimePerson.contains(item.getId())){
                log.info("不是今天第一次到店，跳过到店次数统计处理");
                continue;
            }

            long mallId = item.getMallId();
            long accountId = item.getAccountId();
            MallDayHistoryArriveCount data = map.computeIfAbsent(mallId, x -> new MallDayHistoryArriveCount());
            data.setMallId(mallId);
            data.setAccountId(accountId);
            data.setCountdate(item.getCountdate());
            data.add(item.getHistoryArrivalCount());

        }

        for (MallDayHistoryArriveCount count : map.values()) {
            boolean exist = mallDayDataExist(count.getCountdate(), count.getMallId());
            if (exist) {
                log.info("更新历史到店次数");
                jdbcTemplate.update(SQL_UPDATE_HISTORY_ARRIVE, count.getOnce(), count.getTwice(), count.getThird(), count.getFourth(), count.getFifth(), count.getMore(), count.getCountdate(), count.getMallId());
            } else {
                log.info("新增历史到店次数");
                jdbcTemplate.update(SQL_INSERT_HISTORY_ARRIVE, count.getMallId(), count.getAccountId(), count.getCountdate(), count.getOnce(), count.getTwice(), count.getThird(), count.getFourth(), count.getFifth(), count.getMore());
            }
        }

        log.info("历史到店次数统计结束");
    }

    /**
     * 通过日期和时间获得历史到店人员personUnid集合的redisKey
     */
    public String getHistoryPersonUnidRedisKey(Date countDate, long mallId) {
        String dateStr = DateUtil.format("yyyyMMdd", countDate);
        return "personUnidSet:" + dateStr + ":" + mallId;
    }

    /**
     * 通过日期和时间获得广场历史到店的key
     */
    public String getHistoryArriveCountRedisKey(Date countDate, long mallId) {
        String dateStr = DateUtil.format("yyyyMMdd", countDate);
        return "historyArriveCount:" + dateStr + ":" + mallId;
    }


    public void loadHistoryArriveCountKey2Redis(Date countDate, long mallId) {
        String key = getHistoryArriveCountRedisKey(countDate, mallId);
        if (!redisTemplate.hasKey(key)) {
            redissonService.lockAndRun(LOCK_HISTORY_ARRIVE,300L,299L,()->{
                if (!redisTemplate.hasKey(key)) {
                    List<MallDayHistoryArriveCount> list = jdbcTemplate.query(SQL_SELECT_HISTORY_ARRIVE, new BeanPropertyRowMapper<>(MallDayHistoryArriveCount.class), countDate, mallId);
                    for (MallDayHistoryArriveCount data : list) {
                        log.info("开始load历史到店次数key值到redis,mallId:{},countDate:{}", data.getMallId(), data.getCountdate());
                        String redisKey = getHistoryArriveCountRedisKey(data.getCountdate(), data.getMallId());
                        BoundHashOperations<String, Object, Object> op = redisTemplate.boundHashOps(redisKey);
                        op.increment("flag", 1);
                        op.expire(1, TimeUnit.DAYS);
                    }
                }
            });
        }
    }

    /**
     * 缓存对应日期的personUnid到数据库
     */
    public void loadMallDayPersonUnid2Redis(Date countDate, long mallId) {
        String redisKey = getHistoryPersonUnidRedisKey(countDate, mallId);
        if (!redisTemplate.hasKey(redisKey)) {
            redissonService.lockAndRun(LOCK_PERSON_UNID,300L,299L,()->{
                if (!redisTemplate.hasKey(redisKey)) {
                    log.info("开始load今日到店人员personUnid,mallId:{},countDate:{}", mallId, countDate);
                    List<String> personUnidList = jdbcTemplate.queryForList("select person_unid from d_face_recognition where countdate=? and mall_id=? and status=1", String.class, countDate, mallId);
                    BoundSetOperations<String, String> setOperations = redisTemplate.boundSetOps(redisKey);
                    personUnidList.forEach(setOperations::add);
                    setOperations.expire(1, TimeUnit.DAYS);
                }
            });
        }
    }

    /**
     * 判断数据库中是否存在该天该广场的历史到店数据
     */
    public boolean mallDayDataExist(Date countDate, long mallId) {
        String key = getHistoryArriveCountRedisKey(countDate, mallId);
        loadHistoryArriveCountKey2Redis(countDate, mallId);

        boolean result = redisTemplate.hasKey(key);
        if (!result) {
            BoundHashOperations<String, Object, Object> ops = redisTemplate.boundHashOps(key);
            ops.increment("flag", 1);
            ops.expire(1, TimeUnit.DAYS);
        }
        return result;
    }
}
