package com.viontech.keliu.batch.item.writer;

import com.viontech.keliu.constants.SystemConstants;
import com.viontech.keliu.model.*;
import com.viontech.keliu.service.HistoryArriveCountService;
import com.viontech.keliu.service.OrgCacheService;
import com.viontech.keliu.service.RedissonService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ItemStreamException;
import org.springframework.batch.item.ItemStreamWriter;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Component("faceRecognitionSta2RedisWriter")
public class FaceRecognitionSta2RedisWriter implements ItemStreamWriter<FaceRecognition> {

	private final Logger logger = LoggerFactory.getLogger(FaceRecognitionSta2RedisWriter.class);

	private static final String GATE_SYNC = "LOCK_Face2Redis_Gate";
	private static final String MALL_SYNC = "LOCK_Face2Redis_Mall";
	private static final String FLOOR_SYNC = "LOCK_Face2Redis_Floor";
	private static final String ZONE_SYNC = "LOCK_Face2Redis_Zone";
	@Resource
	private RedissonService redissonService;

    private ThreadLocal<ExecutionContext> executionContextThreadLocal = new ThreadLocal<>();


	@Resource
	private JdbcTemplate jdbcTemplate;

	@Resource
	private RedisTemplate redisTemplate;

	@Resource
	private OrgCacheService orgCacheService;

	@Resource
	private HistoryArriveCountService historyArrivalCountService;

	@Override
	public void open(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
	}

	@Override
	public void update(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
	}

	@Override
	public void close() throws ItemStreamException {
        executionContextThreadLocal.remove();
	}

	@Override
	public void write(List<? extends FaceRecognition> items) throws Exception {
		Map<String,Channel> channelMap = orgCacheService.getChannelMap();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
		for (FaceRecognition item : items) {
			Channel channel = channelMap.get(item.getChannelSerialnum());
			if(channel == null){
				logger.warn("设备{}不存在",item.getChannelSerialnum());
				continue;
			}
			Long gateId = channel.getGateId();
			if(gateId == null){
				logger.warn("设备{}没有绑定监控点",item.getChannelSerialnum());
				continue;
			}
			String dateStr = sdf.format(item.getCountdate());
			Map<String,List<ZoneGate>> gate2ZoneMap = orgCacheService.getGate2ZoneMap();
			Map<String,Zone> zoneMap = orgCacheService.getZoneMap();
			Map<String,Long> gate2MallMap = (Map<String, Long>) orgCacheService.getGate2MallMap();
			Long mallId = gate2MallMap.get(String.valueOf(channel.getGateId()));
			//gate处理
			Map<String,Object> map = new HashMap<String, Object>();
			if(item.getMallId()!=null) {
				map.put("mallId", item.getMallId());
			}else{
				logger.debug("{}无法从处理后的组织机构中获取到MallId,改为从Gate中获取",item.getChannelSerialnum());
				map.put("mallId", mallId);
			}
			map.put("accountId", item.getAccountId());
			map.put("countDate", item.getCountdate());
			map.put("gateId", gateId);

			StringBuilder personSb= new StringBuilder();
			personSb.append("newOrRegularPersonUnid:").append("day:").append("mall:").append(dateStr).append(":").append(mallId);
			String personUnidKey = personSb.toString();

			StringBuilder sb= new StringBuilder();
			String faceStaKey;
			BoundHashOperations boundHashOps;

            ExecutionContext executionContext = executionContextThreadLocal.get();
            Set<Long> firstTimePerson  = (Set<Long>) executionContext.get("firstTimePerson");
            boolean isRepeat = true;
			if(firstTimePerson != null && firstTimePerson.contains(item.getId())){
				isRepeat = false;
			}
			 historyArrivalCountService.putHistoryArriveCount(item);
			//mall 处理
			if(mallId != null&&item.getDirection()== SystemConstants.FACE_DIRECTION_IN){//商场的人脸统计只统计进
				sb.setLength(0);
				sb.append("faceSta:").append("day:").append("mall:").append(dateStr).append(":").append(mallId);
				faceStaKey = sb.toString();
				boundHashOps = redisTemplate.boundHashOps(faceStaKey);
				if (!redisTemplate.expire(faceStaKey,2,TimeUnit.HOURS)) {
					String finalFaceStaKeyMall = faceStaKey;
					BoundHashOperations finalBoundHashOpsMall = boundHashOps;
					redissonService.lockAndRun(MALL_SYNC,300l,299l,()->{
						if (!redisTemplate.hasKey(finalFaceStaKeyMall)) {
							logger.info("开始加载 <商场> <天级> 统计缓存 key:{}", finalFaceStaKeyMall);
							loadMallDayFaceSta(mallId, item.getCountdate(), finalBoundHashOpsMall);
						}
					});
				}
				incrementValue(boundHashOps,item,isRepeat);
				boundHashOps.putAll(map);
			}else{
				if(mallId == null){
					logger.debug("无法统计到商场人脸中，商场ID无法获取,通道号为："+item.getChannelSerialnum());
				}else if(item.getDirection()!= SystemConstants.FACE_DIRECTION_IN){
					logger.debug("无法统计到商场人脸中，方向不是进,通道号为："+item.getChannelSerialnum());
				}
			}
			//floor处理

			sb.setLength(0);
			sb.append("faceSta:").append("day:").append("gate:").append(dateStr).append(":").append(gateId);
			faceStaKey = sb.toString();
			boundHashOps = redisTemplate.boundHashOps(faceStaKey);
			if (!redisTemplate.expire(faceStaKey,2,TimeUnit.HOURS)) {
				String finalFaceStaKeyGate = faceStaKey;
				BoundHashOperations finalBoundHashOpsGate = boundHashOps;
				redissonService.lockAndRun(GATE_SYNC,300l,299l,()->{
					if (!redisTemplate.hasKey(finalFaceStaKeyGate)) {
						logger.info("开始加载 <监控点> <天级> 统计缓存 key:{}", finalFaceStaKeyGate);
						loadGateDayFaceSta(mallId,gateId, item.getCountdate(), finalBoundHashOpsGate);
					}
				});
			}

			incrementValue(boundHashOps,item,isRepeat);
			boundHashOps.putAll(map);

			Map<String,List<FloorGate>> floor2ZoneMap = (Map<String, List<FloorGate>>) orgCacheService.getGate2FloorMap();
			List<FloorGate> floorGates = floor2ZoneMap.get(String.valueOf(channel.getGateId()));
			if(floorGates != null && floorGates.size() > 0){
				for (FloorGate floorGate : floorGates) {
					long floorId = floorGate.getFloorId();
					map.put("floorId", floorId);
					int type = floorGate.getType();
					if(type == SystemConstants.ORG_GATE_BIND_TYPE_IN){//如果是正向 只有进的数据统计  如果是反向  那么只有出的数据进行统计
						sb.setLength(0);
						sb.append("faceSta:").append("day:").append("floor:").append(dateStr).append(":").append(floorId);
						faceStaKey = sb.toString();
						boundHashOps = redisTemplate.boundHashOps(faceStaKey);
						if (!redisTemplate.expire(faceStaKey,2,TimeUnit.HOURS)) {
							String finalFaceStaKeyFloor = faceStaKey;
							BoundHashOperations finalBoundHashOpsFloor = boundHashOps;
							redissonService.lockAndRun(FLOOR_SYNC,300L,299L,()->{
								if (!redisTemplate.hasKey(finalFaceStaKeyFloor)) {
									logger.info("开始加载 <楼层> <天级> 统计缓存 key:{}", finalFaceStaKeyFloor);
									loadFloorDayFaceSta(mallId,floorId, item.getCountdate(), finalBoundHashOpsFloor);
								}
							});
						}
						incrementValue(boundHashOps,item,isRepeat);
						boundHashOps.putAll(map);
					}
				}
			}


			//zone处理
			List<ZoneGate> zoneGates = gate2ZoneMap.get(String.valueOf(gateId));
			if(zoneGates != null && zoneGates.size() > 0){
				for (ZoneGate zoneGate : zoneGates) {
					int type = zoneGate.getType();
					if(type == SystemConstants.ORG_GATE_BIND_TYPE_IN){//如果是正向 只有进的数据统计  如果是反向  那么只有出的数据进行统计
						long zoneId = zoneGate.getZoneId();
						Zone zone = zoneMap.get(String.valueOf(zoneId));
						if(zone == null){
							logger.info("zoneId {} null , zoneMap size is {}",zoneId,zoneMap.size());
							continue;
						}
						sb.setLength(0);
						sb.append("faceSta:").append("day:").append("zone:").append(dateStr).append(":").append(zoneId);
						faceStaKey = sb.toString();

						boundHashOps = redisTemplate.boundHashOps(faceStaKey);
						if (!redisTemplate.expire(faceStaKey,2,TimeUnit.HOURS)) {
							String finalFaceStaKeyZone = faceStaKey;
							BoundHashOperations finalBoundHashOpsZone = boundHashOps;
							redissonService.lockAndRun(ZONE_SYNC,300L,299L,()->{
								if (!redisTemplate.hasKey(finalFaceStaKeyZone)) {
									logger.info("开始加载 <区域> <天级> 统计缓存 key:{}", finalFaceStaKeyZone);
									loadZoneDayFaceSta(mallId,zoneId, item.getCountdate(), finalBoundHashOpsZone);
								}
							});
						}
						map.put("zoneId", zoneId);
						map.put("floorId", zone.getFloorId());
						incrementValue(boundHashOps,item,isRepeat);
						boundHashOps.putAll(map);
					}
				}
			}

		}
	}

	private void loadMallDayFaceSta(Long mallId, Date countDate, BoundHashOperations boundHashOps) {
		jdbcTemplate.query("SELECT *  FROM d_mall_day_face_recognition_sta WHERE countdate=? and mall_id=? LIMIT 1", new Object[]{countDate,mallId}, getRowCallbackHandler(boundHashOps));
	}
	private void loadGateDayFaceSta(Long mallId,Long gateId, Date countDate, BoundHashOperations boundHashOps){
		jdbcTemplate.query("SELECT *  FROM d_gate_day_face_recognition_sta WHERE countdate=? and mall_id=? AND gate_id=?  LIMIT 1", new Object[]{countDate,mallId,gateId}, getRowCallbackHandler(boundHashOps));
	}
	private void loadFloorDayFaceSta(Long mallId,Long floorId, Date countDate, BoundHashOperations boundHashOps) {
		jdbcTemplate.query("SELECT *  FROM d_floor_day_face_recognition_sta WHERE countdate=? and mall_id=? AND floor_id=?  LIMIT 1", new Object[]{countDate,mallId,floorId}, getRowCallbackHandler(boundHashOps));
	}
	private void loadZoneDayFaceSta(Long mallId,Long zoneId, Date countDate, BoundHashOperations boundHashOps) {
		jdbcTemplate.query("SELECT *  FROM d_zone_day_face_recognition_sta WHERE countdate=? and mall_id=? AND zone_id=? LIMIT 1", new Object[]{countDate,mallId,zoneId}, getRowCallbackHandler(boundHashOps));
	}
	private RowCallbackHandler getRowCallbackHandler(BoundHashOperations boundHashOps) {
		return rs -> {
			boundHashOps.increment("personMantime", (long) rs.getInt("person_mantime"));
			boundHashOps.increment("personCount", (long) rs.getInt("person_count"));
			boundHashOps.increment("customMantime", (long) rs.getInt("custom_mantime"));
			boundHashOps.increment("customCount", (long) rs.getInt("custom_count"));
			boundHashOps.increment("staffCount", (long) rs.getInt("staff_count"));
			boundHashOps.increment("staffMantime", (long) rs.getInt("staff_mantime"));
			boundHashOps.increment("maleMantime", (long) rs.getInt("male_mantime"));
			boundHashOps.increment("maleCount", (long) rs.getInt("male_count"));
			boundHashOps.increment("femaleMantime", (long) rs.getInt("female_mantime"));
			boundHashOps.increment("femaleCount", (long) rs.getInt("female_count"));

			boundHashOps.increment("newCustomCount", (long) rs.getInt("new_custom_count"));
			boundHashOps.increment("newMaleCount", (long) rs.getInt("new_male_count"));
			boundHashOps.increment("newFemaleCount", (long) rs.getInt("new_female_count"));

			boundHashOps.increment("regularCustomCount", (long) rs.getInt("regular_custom_count"));
			boundHashOps.increment("regularMaleCount", (long) rs.getInt("regular_male_count"));
			boundHashOps.increment("regularFemaleCount", (long) rs.getInt("regular_female_count"));



			if(rs.getString("male_stage")!=null) {
				String[] maleStage = rs.getString("male_stage").split(",");

				for (int i = 0; i < maleStage.length; i++) {
					int count = Integer.parseInt(maleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("maleDetail" + i, (long) count);
					}
				}
			}
			if(rs.getString("female_stage")!=null) {
				String[] femaleStage = rs.getString("female_stage").split(",");
				for (int i = 0; i < femaleStage.length; i++) {
					int count = Integer.parseInt(femaleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("femaleDetail" + i, (long) count);
					}
				}
			}


			if(rs.getString("new_male_stage")!=null) {
				String[] maleStage = rs.getString("new_male_stage").split(",");

				for (int i = 0; i < maleStage.length; i++) {
					int count = Integer.parseInt(maleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("newMaleDetail" + i, (long) count);
					}
				}
			}
			if(rs.getString("new_female_stage")!=null) {
				String[] femaleStage = rs.getString("new_female_stage").split(",");
				for (int i = 0; i < femaleStage.length; i++) {
					int count = Integer.parseInt(femaleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("newFemaleDetail" + i, (long) count);
					}
				}
			}
			if(rs.getString("regular_male_stage")!=null) {
				String[] maleStage = rs.getString("regular_male_stage").split(",");

				for (int i = 0; i < maleStage.length; i++) {
					int count = Integer.parseInt(maleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("regularMaleDetail" + i, (long) count);
					}
				}
			}
			if(rs.getString("regular_female_stage")!=null) {
				String[] femaleStage = rs.getString("regular_female_stage").split(",");
				for (int i = 0; i < femaleStage.length; i++) {
					int count = Integer.parseInt(femaleStage[i].trim());
					if (count > 0) {
						boundHashOps.increment("regularFemaleDetail" + i, (long) count);
					}
				}
			}

		};
	}


	
	private void incrementValue(BoundHashOperations boundHashOps, FaceRecognition faceRecognition,boolean repeatPersonUnid){
		if(repeatPersonUnid){
			return ;
		}
		if(faceRecognition.getPersonType()!=null&&faceRecognition.getPersonType().intValue() == SystemConstants.FACE_TYPE_STAFF) {//如果是顾客 除了店员都是顾客
			return;
		}
		String keyPrefix = "new";//默认新顾客
		Short historyArrivalCount = faceRecognition.getHistoryArrivalCount();
		if(historyArrivalCount != null && historyArrivalCount >0){//老顾客
			keyPrefix = "regular";

		}
		String genderStr = faceRecognition.getGender()== SystemConstants.FACE_GENDER_MALE?(keyPrefix+"Male"):(keyPrefix+"Female");//1男 其他女
		boundHashOps.increment(keyPrefix+"CustomCount", 1L);//新老顾客人数+1
		String countKey = genderStr+"Count";
		boundHashOps.increment(countKey, 1L);// 新老男顾客或女顾客人数+1

		String infoKey = genderStr+"Detail"+faceRecognition.getAge();// newMaleDetail10  新 男性 10岁的顾客
		boundHashOps.increment(infoKey,1L);//新老顾客男女性别年龄对应的人数+1
	}

}

