package com.viontech.keliu.batch.item.processor;

import com.viontech.keliu.model.*;
import com.viontech.keliu.service.ConfigParamService;
import com.viontech.keliu.service.CustomMatchService;
import com.viontech.keliu.service.HistoryArriveCountService;
import com.viontech.keliu.service.OrgCacheService;
import com.viontech.keliu.util.ConfigUtil;
import com.viontech.keliu.util.DateUtil;
import com.viontech.keliu.websocket.AlgApiClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.batch.item.ItemProcessor;
import org.springframework.batch.item.ItemStream;
import org.springframework.batch.item.ItemStreamException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;

@Component("historyArriveProcessor")
public class HistoryArriveProcessor implements ItemProcessor<FaceRecognition, FaceRecognition>, ItemStream {

    private final Logger logger = LoggerFactory.getLogger(HistoryArriveProcessor.class);

    @Value("${match.score.camera:1.5}")
    private Float matchScoreCamera;
    @Value("${ws.url:}")
    private String url;

    @Autowired(required = false)
    private AlgApiClient algApiClientComparison;

    @Value("${match.score.custom}")
    private Integer matchScoreCustom;
    @Resource
    private OrgCacheService orgCacheService;
    @Resource
    private ConfigParamService configParamService;
    @Resource
    private CustomMatchService customMatchService;
    @Resource
    private HistoryArriveCountService historyArrivalCountService;

    private ThreadLocal<ExecutionContext> executionContextThreadLocal = new ThreadLocal<>();


    @Override
    public FaceRecognition process(FaceRecognition faceRecognition){
        // 状态及默认新顾客赋值
        faceRecognition.setStatus((short) 1);
        faceRecognition.setHistoryArrivalCount((short) 0);
        if (algApiClientComparison == null) {
            logger.debug("该系统未配置新老客户比对服务器，跳过");
            return faceRecognition;
        }
        Long mallId = faceRecognition.getMallId();
        if (mallId == null) {
            logger.warn("该条人脸数据的mallId为空，跳过");
            return faceRecognition;
        }

        Map<String, Mall> mallMap = orgCacheService.getMallMap();
        Mall mall = mallMap.get(String.valueOf(mallId));
        if (mall == null) {
            logger.warn("当前数据找不到对应的商场 {} 为空,跳过", mallId);
            return faceRecognition;
        }

        ExecutionContext executionContext = executionContextThreadLocal.get();
        Map<Long, ConfigParam> configParamMap = (Map<Long, ConfigParam>) executionContext.get("configMap");
        ConfigParam configParam = ConfigUtil.getConfigParam(mallId, configParamMap);
        String newOrRegular = configParam.getNewOrRegular();
        int matchDay = 0;
        try {
            matchDay = Integer.parseInt(newOrRegular);
        } catch (Exception e) {}

        if (matchDay == 0) {
            logger.debug("商场 {} 关闭了新老顾客功能,跳过", mall.getName());
            return faceRecognition;
        }

        logger.debug("商场 {} 开启了新老顾客功能,匹配历史 {} 天的历史顾客", faceRecognition.getMallId(), newOrRegular);

        try {
            //如果一个person已经是今天第二次进行新老顾客比对 那么他的比对结果应该与redis中的一致，所以不再执行比对，直接更新他的历史到店次数即可，历史到店记录与第一次一致
            boolean success = historyArrivalCountService.hasHistoryArriveCount(faceRecognition);

            if (!success) {
                return faceRecognition;
            }

            Set<Long> firstTimePerson  = (Set<Long>) executionContext.get("firstTimePerson");
            if(firstTimePerson == null){
                firstTimePerson = new HashSet<>();
                executionContext.put("firstTimePerson",firstTimePerson);
            }

            firstTimePerson.add(faceRecognition.getId());
            //开始进行比对
            MatchParam matchParam = new MatchParam();
            matchParam.setMallId(mallId);
            List<Date> matchDates = buildMatchDates(faceRecognition.getCountdate(), matchDay);
            matchParam.setDates(matchDates);
            matchParam.setAppend(false);
            matchParam.setFaceScoerThreshold(matchScoreCamera);
            matchParam.setMatchThreshold(matchScoreCustom);
            matchParam.setSubPool(configParam.getSubPool()==null?true:"1".equals(configParam.getSubPool()));
            Person person = customMatchService.buildPerson(faceRecognition);
            List<Person> matchPersons = customMatchService.match(person, matchParam);
            if (matchPersons == null || matchPersons.size() <= 0) {
                return faceRecognition;
            }
            // 历史到店次数就是匹配到的顾客数量
            faceRecognition.setHistoryArrivalCount((short) matchPersons.size());
            historyArrivalCountService.putHistoryArriveCount(faceRecognition);
            List<HistoryArriveRecord> hisArrivalRecords = new ArrayList<>();

            // 开始构建人员历史到店记录
            for (Person matchPerson : matchPersons) {
                HistoryArriveRecord historyArrivalRecord = new HistoryArriveRecord();
                historyArrivalRecord.setPersonUnid(faceRecognition.getPersonUnid());
                historyArrivalRecord.setMallId(faceRecognition.getMallId());
                historyArrivalRecord.setAccountId(faceRecognition.getAccountId());
                historyArrivalRecord.setCountdate(faceRecognition.getCountdate());
                String personPoolId = matchPerson.getPersonPoolId();
                String[] personPoolSplits = personPoolId.split("_");
                Date hisArrivalDate = DateUtil.parse(DateUtil.FORMAT_SHORT, personPoolSplits[2]);
                historyArrivalRecord.setHistoryArriveDate(hisArrivalDate);
                historyArrivalRecord.setHistoryArrivePersonUnid(matchPerson.getPersonId());

                hisArrivalRecords.add(historyArrivalRecord);
            }
            faceRecognition.setHistoryArrivalRecords(hisArrivalRecords);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return faceRecognition;
    }

    private List<Date> buildMatchDates(Date countDate, Integer matchDay) {
        List<Date> matchDates = new ArrayList<>();
        Date matchDate;
        for (int i = 1; i <= matchDay; i++) {
            matchDate = DateUtil.addDays(countDate, -i);
            matchDates.add(matchDate);
        }
        return matchDates;
    }

    @Override
    public void open(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
        Map<Long, ConfigParam> configParamMap = configParamService.selectAll();
        executionContext.put("configMap", configParamMap);
    }

    @Override
    public void update(ExecutionContext executionContext) throws ItemStreamException {
        executionContextThreadLocal.set(executionContext);
        Map<Long, ConfigParam> configParamMap = configParamService.selectAll();
        executionContext.put("configMap", configParamMap);
    }

    @Override
    public void close() throws ItemStreamException {
        ExecutionContext executionContext = executionContextThreadLocal.get();
        executionContext.remove("configMap");
        executionContextThreadLocal.remove();
    }



}
