package com.viontech.keliu.batch.configuration;

import com.viontech.keliu.batch.item.reader.RedisPopReader;
import com.viontech.keliu.batch.item.writer.HistoryArriveCountUpdateWriter;
import com.viontech.keliu.batch.item.writer.HistoryArriveRecordWriter;
import com.viontech.keliu.batch.listener.JobRestartListener;
import com.viontech.keliu.constants.RedisKeyConstants;
import com.viontech.keliu.content.FaceDataContent;
import com.viontech.keliu.model.FaceRecognition;
import com.viontech.keliu.model.HistoryArriveRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.Job;
import org.springframework.batch.core.Step;
import org.springframework.batch.core.configuration.annotation.JobBuilderFactory;
import org.springframework.batch.core.configuration.annotation.StepBuilderFactory;
import org.springframework.batch.core.launch.support.RunIdIncrementer;
import org.springframework.batch.core.repository.JobRepository;
import org.springframework.batch.item.ItemProcessor;
import org.springframework.batch.item.ItemReader;
import org.springframework.batch.item.ItemWriter;
import org.springframework.batch.item.database.BeanPropertyItemSqlParameterSourceProvider;
import org.springframework.batch.item.support.CompositeItemWriter;
import org.springframework.batch.repeat.RepeatContext;
import org.springframework.batch.repeat.RepeatStatus;
import org.springframework.batch.repeat.policy.CompletionPolicySupport;
import org.springframework.batch.repeat.support.RepeatTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.core.RedisTemplate;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.List;

@Configuration
public class HistoryArriveJobConfiguration {

	private final Logger logger = LoggerFactory.getLogger(HistoryArriveJobConfiguration.class);

	@Autowired
	private JobBuilderFactory jobBuilderFactory;

	@Autowired
	private StepBuilderFactory stepBuilderFactory;

	@Autowired


	@Value("${spring.batch.job.chunkSize:1}")
	private int chunkSize;

	/* JOB定义   START */
	/**
	 * 人脸抓拍记录（历史到店情况）Job
	 * @param jobRepository
	 * @return
	 */
	@Bean(name = "historyArriveJob")
	public Job historyArriveJob(JobRepository jobRepository, Step historyArriveStep, JobRestartListener jobRestartListener) {
		return this.jobBuilderFactory.get("historyArriveJob")
				.listener(jobRestartListener)
				.repository(jobRepository).incrementer(new RunIdIncrementer()).start(historyArriveStep).build();
	}

	@Bean(name = "historyArriveStep")
	public Step historyArriveStep(ItemReader<FaceRecognition> faceRecognitonReader4Redis,
							  ItemProcessor historyArriveProcessor,
							  ItemWriter historyArriveCountUpdateWriter,
							  ItemWriter faceRecognitionSta2RedisWriter,
							  ItemWriter historyArriveRecordWriter,
							  ItemWriter mallDayHistoryArriveCountWriter) {
		RepeatTemplate stepOperations = new RepeatTemplate();
		stepOperations.setCompletionPolicy(new CompletionPolicySupport() {
			@Override
			public boolean isComplete(RepeatContext context, RepeatStatus result) {
				return false;
			}

			@Override
			public boolean isComplete(RepeatContext context) {
				return false;
			}

		});

		CompositeItemWriter<List<FaceRecognition>> compositeItemWriter  = new CompositeItemWriter<List<FaceRecognition>>();
		List delegates = new ArrayList();
		delegates.add(faceRecognitionSta2RedisWriter);//写redis 这个必须放在上边，放在后边 事务会影响数据。
		delegates.add(historyArriveCountUpdateWriter);// 更新抓拍记录 历史到店次数
		delegates.add(historyArriveRecordWriter); // 插入历史到店详情
		delegates.add(mallDayHistoryArriveCountWriter);//历史到店次数统计
		compositeItemWriter.setDelegates(delegates);
		return this.stepBuilderFactory.get("hisArriveStep").
				<FaceRecognition,FaceRecognition>chunk(chunkSize).reader(faceRecognitonReader4Redis).processor(historyArriveProcessor).writer(compositeItemWriter).stepOperations(stepOperations).build();
	}


	/**
	 * redis客流数据读取器
	 * @return
	 */
	@Bean("faceRecognitonReader4Redis")
	public ItemReader faceRecognitonReader4Redis(RedisTemplate redisTemplate) {
		RedisPopReader<FaceDataContent> redisPopReader = new RedisPopReader<FaceDataContent>();
		redisPopReader.setRedisTemplate(redisTemplate);
		redisPopReader.setKey(RedisKeyConstants.NEW_REGULAR_CUSTOM_PROCESS_LIST);
		return redisPopReader;
	}


	/**
	 * 用于更新人脸抓拍表的历史到店次数
	 * @param dataSource
	 * @return
	 */
	@Bean(name="historyArriveCountUpdateWriter")
	public ItemWriter historyArriveCountUpdateWriter(DataSource dataSource){
		HistoryArriveCountUpdateWriter historyArriveCountUpdateWriter = new HistoryArriveCountUpdateWriter(dataSource);
		historyArriveCountUpdateWriter.setItemSqlParameterSourceProvider(new BeanPropertyItemSqlParameterSourceProvider<FaceRecognition>());
		return historyArriveCountUpdateWriter;
	}

	/**
	 * 用于插入历史到店记录到 d_his_arr
	 * @param dataSource
	 * @return
	 */
	@Bean(name="historyArriveRecordWriter")
	public ItemWriter HisArriveRecordWriter(DataSource dataSource){
		HistoryArriveRecordWriter historyArriveRecordWriter = new HistoryArriveRecordWriter(dataSource);
		historyArriveRecordWriter.setItemSqlParameterSourceProvider(new BeanPropertyItemSqlParameterSourceProvider<HistoryArriveRecord>());
		return historyArriveRecordWriter;
	}
}
