package com.viontech.keliu.match;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.viontech.keliu.model.*;
import com.viontech.keliu.service.CreateFeaturePoolService;
import com.viontech.keliu.websocket.AlgApiClient;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by 苏曼 on 2019/7/16.
 */
public abstract class AbstractMatchService implements MatchService {

    private final Logger logger = LoggerFactory.getLogger(AbstractMatchService.class);
    @Autowired
    protected ObjectMapper objectMapper;
    @Autowired(required = false)
    private AlgApiClient algApiClientComparison;
    @Autowired
    private CreateFeaturePoolService createFeaturePoolService;

    /**
     * 人员匹配
     *
     * @param person     需要匹配的人
     * @param matchParam 匹配参数
     *
     * @return 匹配结果
     */
    @Override

    public List<Person> match(Person person, MatchParam matchParam) throws Exception {
        Map<String, String> poolNameMap = buildPoolName(matchParam); // 构建池名

        if (poolNameMap == null || poolNameMap.size() == 0) {
            logger.error("Bug 构建匹配库不存在！");
            return null;
        }
        poolNameMap = poolNameMap.entrySet().stream().collect(Collectors.toMap(x -> x.getKey().toLowerCase(), Map.Entry::getValue, (x, y) -> x));

        buildPerson(person);
        List<BodyFeature> originalBodyFeatures = person.getBodyFeatures();
        // 如果不需要匹配人体 ,那么将人体先置空，然后匹配完成之后再设置回去
        if (!matchParam.isMatchBody()) {
            person.setBodyFeatures(null);
        }
        // 开始匹配人员
        CompletableFuture<JSONObject> responseFuture;
        List<String> poolNames = new ArrayList<>(poolNameMap.keySet());
        if (poolNames.size() == 1) {
            responseFuture = algApiClientComparison.matchPerson(2, person, poolNames.get(0), Collections.emptyList(), new HashMap<>());
        } else {
            responseFuture = algApiClientComparison.matchHistoryPerson(2, person, poolNames, new HashMap<>());
        }

        JSONObject response = responseFuture.get(60, TimeUnit.SECONDS);
        Result result = objectMapper.readValue(response.toString(), Result.class);
        // 匹配人员结束
        if (result == null) {
            logger.info("人员匹配结束 没有结果 直接跳过");
            return null;
        }
        // 处理特征池不存在的逻辑， 建池 然后重新比对
        Integer match = result.getMatch();
        Integer errCode = result.getErrCode();
        logger.info(response.toString());
        if (errCode != null && errCode == 4 && match == 0) {//特征池不存在，创建特征池。然后再处理一遍
            poolNameMap.forEach((poolName, type) -> buildPool(poolName, type, matchParam));
            return match(person, matchParam);
        }

        List<Pool> personPoolStatus = result.getPersonPoolStatus();
        if (personPoolStatus != null && personPoolStatus.size() > 0) {
            boolean hasNewPool = false;
            for (Pool pool : personPoolStatus) {
                int status = pool.getStatus();
                String poolName = pool.getPersonPoolId();
                if (status == 1) {//处理未建池，建池并重新匹配封装
                    buildPool(poolName, poolNameMap.get(poolName), matchParam);
                    hasNewPool = true;
                }
            }
            if (hasNewPool) {
                return match(person, matchParam);
            }
        }

        // 如果需要追加 且 匹配不到人
        if (matchParam.isAppend() && !isMatchedPerson(result)) {
            appendPerson2Pool(person, poolNames);
            return null;
        }
        List<Person> matchResult = analysisMatchPerson(person, poolNames, result.getMatchPersons(), result.getMatchBodies(), matchParam);
        person.setBodyFeatures(originalBodyFeatures);
        return matchResult;

    }

    private boolean isMatchedPerson(Result result) {
        if (result.getSuccess() == 0) {
            return false;
        }
        if (result.getMatch() == 0) {
            return false;
        }
        if (result.getMatchPersons() != null && result.getMatchPersons().size() > 0) { // 如果匹配到人 那么返回true
            return true;
        }
        // 如果匹配到人体 那么返回true
        return result.getMatchBodies() != null && result.getMatchBodies().size() > 0;
    }


    private List<Person> analysisMatchPerson(Person person, List<String> poolNames, List<Person> matchedPersons, List<Person> matchedBodies, MatchParam matchParam) {
        List<Person> result = new ArrayList<>();
        if (matchedPersons == null || matchedPersons.isEmpty()) {
            return result;
        }
        Map<String, Person> matchedMap = new HashMap<>();

        Map<String, Person> matchedBodyMap = new HashMap<>();
        // 处理匹配结果 找出来每个池里分数最高的
        matchedPersons.forEach(matchedPerson -> {
            Integer matchedScore = matchedPerson.getScore();
            logger.info("[{}]-[{}]人脸匹配得分[{}]", person.getPersonId(), matchedPerson.getPersonId(), matchedScore);
            if (matchedScore >= matchParam.getMatchThreshold()) {//如果匹配到人
                Person p = matchedMap.get(matchedPerson.getPersonPoolId());
                // 将同一个池里的匹配结果修改为分数最高的那个
                if (p == null || matchedScore > p.getScore()) {
                    matchedMap.put(matchedPerson.getPersonPoolId(), matchedPerson);
                }
            }
        });
        // 人脸匹配完成之后有两种情况，一种是能识别出来，那么需要将特征追加给识别者
        // 一种是无法识别到，那么需要新建一个person放入池中，等待被匹配
        // 由于这儿是一个人到多个池子中去匹配，那么每个池都要执行上面的操作

        Float faceScore = matchParam.getFaceScore();
        // 如果需要追加
        if (matchParam.isAppend()) {
            // 循环所有去匹配的人员池
            poolNames.forEach(poolName -> {
                Person matchedPerson = matchedMap.get(poolName);
                // 如果没有匹配到人 追加新人
                // 如果匹配到人了，如果分数满足特征追加条件，追加特征
                if (matchedPerson == null) {
                    logger.info("追加新人[{}]到[{}]-图片质量分:{}", person.getPersonId(), poolName, faceScore);
                    appendPerson2Pool(person, poolName);
                } else if (matchedPerson.getScore() >= matchParam.getAppendThreshold()) { // 人脸分值大于一定程度才追加特征
                    logger.info("追加优质识别结果 {} 到 {}({}) -匹配得分:{},图片质量分:{}", person.getPersonId(), poolName, matchedPerson.getPersonId(), matchedPerson.getScore(), faceScore);
                    // 保存原来的unid 追加用户到池中 再改回来
                    String originalPersonId = person.getPersonId();
                    person.setPersonId(matchedPerson.getPersonId());
                    appendPerson2Pool(person, poolName);
                    person.setPersonId(originalPersonId);
                }
            });
        }

        result.addAll(matchedMap.values());
        // 人体只可识别 不可追加 因为人会换衣服
        if (matchParam.isMatchBody()) { // 如果匹配人体 且 没有通过人脸识别出来结果 那么执行获取人体比对信息
            if (matchedBodies == null) {
                return result;
            }
            matchedBodies.forEach(matchedBody -> {
                Integer score = matchedBody.getScore();
                logger.info("{} 人体匹配得分{}", matchedBody.getPersonId(), score);
                if (score > matchParam.getMatchBodyThreshold()) {//如果是通过人体识别成的一个人 不追加结果到池中
                    Person p = matchedBodyMap.get(matchedBody.getPersonPoolId());
                    // 将同一个池里的匹配结果修改为分数最高的那个
                    if (p == null || score > p.getScore()) {
                        matchedBodyMap.put(matchedBody.getPersonPoolId(), matchedBody);
                    }
                }
            });

            //默认人脸结果为准 如果没有人脸结果 那么使用人体结果

            matchedBodyMap.forEach((poolName, p) -> {
                if (matchedMap.get(poolName) == null) {
                    result.add(p);
                }
            });
        }

        return result;

    }

    /**
     * 构建特征池
     *
     * @param poolName   特征池名称
     * @param type       池类型
     * @param matchParam 匹配参数
     */
    protected void buildPool(String poolName, String type, MatchParam matchParam) {
        JSONObject jsonObject = createFeaturePoolService.createFeaturePool(poolName);
        if (jsonObject == null) {
            logger.warn("特征库:{}创建异常", poolName);
            return;
        } else if (jsonObject.getInt("errCode") == 20) {// 特征库已存在：20
            logger.warn("特征库：{}已存在，跳过创建增加人员步骤", poolName);
            return;
        }
        List<Person> persons = getPersons(matchParam.getMallId(), type, matchParam);
        int size = 0;
        for (Person person : persons) {
            if (!buildPerson(person)) {
                continue;
            }
            appendPerson2Pool(person, poolName);
            size++;

        }

        logger.info("创建人员池{} 完成，添加人数{}", poolName, size);
    }

    /**
     * 构建用户,构建特征
     *
     * @param person 人员
     *
     * @return 是否成功
     */
    private boolean buildPerson(Person person) {
        boolean result = false;
        try {
            List<FaceFeature> faceFeatures = person.getFaceFeatures();
            if (faceFeatures != null && !faceFeatures.isEmpty()) {
                List<FaceFeature> faceFeaturesNew = faceFeatures.stream().filter(this::buildFaceFeature).collect(Collectors.toList());
                if (!faceFeaturesNew.isEmpty()) {
                    result = true;
                    person.setFaceFeatures(faceFeaturesNew);
                }

            }
            List<BodyFeature> bodyFeatures = person.getBodyFeatures();
            if (bodyFeatures != null && !bodyFeatures.isEmpty()) {
                List<BodyFeature> bodyFeaturesNew = bodyFeatures.stream().filter(this::buildBodyFeature).collect(Collectors.toList());
                if (!bodyFeaturesNew.isEmpty()) {
                    result = true;
                    person.setBodyFeatures(bodyFeaturesNew);
                }

            }
        } catch (Exception e) {
            logger.error("异常！！", e);
        }
        return result;
    }

    /**
     * 构建人脸特征
     * 如果特征已存在则跳过
     * 不存在则尝试通过获取特征方法{@link #readFeature(String)} 获取
     * 如果获取失败则尝试重新提取特征 {@link #reExtractFaceFeature(String)}
     * 之后通过{@link #buildFeature(String)} 构建特征并赋给特征对象
     *
     * @param faceFeature 人脸特征
     *
     * @return 是否构建
     */
    private boolean buildFaceFeature(FaceFeature faceFeature) {
        // 如果特征已经存在跳过
        if (faceFeature.getFeature() != null && faceFeature.getFeature().length > 0) {
            return true;
        }
        // 如果特征不存在 尝试获读取
        String featureStr = readFeature(faceFeature.getPicName());
        // 如果特征读取失败 尝试重新提取
        if (featureStr == null) {
            featureStr = reExtractFaceFeature(faceFeature.getPicName());
            if (featureStr == null) {
                return false;
            }
        }
        Double[] featureArr = buildFeature(featureStr);
        faceFeature.setFeature(featureArr);
        return true;
    }

    /**
     * 构建人体特征
     * 如果特征已存在则跳过
     * 不存在则尝试通过获取特征方法{@link #readFeature(String)} 获取
     * 如果获取失败则尝试重新提取特征 {@link #reExtractBodyFeature(String)}
     * 之后通过{@link #buildFeature(String)} 构建特征并赋给特征包装类对象
     *
     * @param bodyFeature 人体特征包装类对象
     *
     * @return 构建是否成功
     */
    private boolean buildBodyFeature(BodyFeature bodyFeature) {
        // 如果特征已经存在跳过
        if (bodyFeature.getFeature() != null && bodyFeature.getFeature().length > 0) {
            return true;
        }
        // 如果特征不存在 尝试获取
        String featureStr = readFeature(bodyFeature.getPicName());
        if (featureStr == null) {
            featureStr = reExtractBodyFeature(bodyFeature.getPicName());
            if (featureStr == null) {
                return false;
            }
        }

        Double[] featureArr = buildFeature(featureStr);
        if (featureArr == null) {
            return false;
        }
        bodyFeature.setFeature(featureArr);
        return true;
    }

    /**
     * 通过特征的Json字符串构建特征
     *
     * @param featureStr Json字符串
     *
     * @return 特征
     */
    private Double[] buildFeature(String featureStr) {
        Feature feature = null;
        try {
            feature = objectMapper.readValue(featureStr, Feature.class);
        } catch (Exception e) {
        }

        if (feature == null) {
            return null;
        }
        if ("face".equals(feature.getType())) { // 如果是人脸特征
            Integer faceType = feature.getFace_type();
            if (faceType != null && faceType != 1) { //不处理 -1 和 0 类型的数据
                return null;
            }
        }
        List<Data> datas = feature.getDatas();
        if (datas == null || datas.isEmpty()) {
            return null;
        }
        Optional<Data> dataResult = datas.stream().filter(item -> "server".equals(item.getType())).findFirst();
        Data data = new Data();
        if (dataResult.isPresent()) {
            data = dataResult.get();
        }
        if (data.getType() == null) {
            return null;
        }
        return data.getData();
    }

    /**
     * 构建特征池时使用,通过广场Id和抓拍类型从数据库获取所有抓拍
     * 通过抓拍构造person对象
     *
     * @param mallId     广场Id
     * @param type       抓拍类型(顾客\店员\人员等)
     * @param matchParam 匹配参数
     *
     * @return 所有抓拍对应的Person集合
     */
    protected abstract List<Person> getPersons(Long mallId, String type, MatchParam matchParam);

    /**
     * 追加人员到多个特征池
     *
     * @param person    需要追加到人员池中的人
     * @param poolNames 需要追加的人员池名称集合
     */
    protected boolean appendPerson2Pool(Person person, List<String> poolNames) {
        poolNames.forEach(poolName -> this.appendPerson2Pool(person, poolName));
        return true;
    }

    /**
     * 追加人员到单个特征池
     *
     * @param person   需要追加到人员池中的人
     * @param poolName 需要追加的人员池名称
     */
    protected boolean appendPerson2Pool(Person person, String poolName) {
        List<Person> persons = new ArrayList<>();
        persons.add(person);
        try {
            algApiClientComparison.modifyPersonPool(poolName, 2, 2, persons, new HashMap<>());
            logger.debug("====添加人员====特征池名称:[{}]====personUnid:[{}]", poolName, person.getPersonId());
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 尝试通过文件名称获取对应的Feature Json字符串
     *
     * @param file 文件名称
     *
     * @return feature Json字符串
     */
    protected abstract String readFeature(String file);

    protected String reExtractFaceFeature(String file) {
        return null;
    }

    protected String reExtractBodyFeature(String file) {
        return null;
    }

    /**
     * 通过参数 {@link MatchParam} 构建需要用来比对的特征池的名称
     *
     * @param matchParam 参数
     *
     * @return Map<String, String> 以特征池名称为key的map对象
     */
    protected abstract Map<String, String> buildPoolName(MatchParam matchParam);

}
