package com.viontech.keliu.dao;


import com.viontech.keliu.entity.DeviceDetails;
import com.viontech.keliu.entity.LogInfo;
import com.viontech.keliu.entity.Operate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * @author 谢明辉
 * @createDate 2019-3-4
 * @description
 */
@Repository
public class DeviceDao {

    private static Logger log = LoggerFactory.getLogger(DeviceDao.class);

    @Resource
    private JdbcTemplate jdbcTemplate;
    @Value("${client.version}")
    private Integer version;

    private static final String SQL_GET_DEVICE_DETAILS = "select t.hardware as type,t.serialnum as deviceSerialNum,t.local_ip as deviceIP,m.name as mallName,m.unid as mallUnid,g.name as gateName,g.unid as gateUnid from (select d.hardware,d.serialnum,d.local_ip,d.mall_id,c.gate_id from b_device d left join b_channel c on d.serialnum=c.device_serialnum where d.status=1) as t LEFT JOIN b_mall as m on t.mall_id=m.id left join b_gate as g on t.gate_id=g.id";

    private static final String SQL_GET_DEVICE_DETAILS_53 = "select o.orgid as mallUnid,o.orgname as mallName,g.gateid as gateUnid,g.gatename as gateName,c.cameraid,t.ip as deviceIP,t.serialnum as deviceSerialNum from tbl_camera_info as c,tbl_gate_info as g,tbl_org_info as o,\n" +
                                                            "(select val.cameraid,vas.ip,vt.serialnum from tbl_va_link as val,tbl_vt_vas as vt,tbl_va_info as va,tbl_vas_info as vas where \n" +
                                                            "val.vaid=va.id and va.vasid=vas.id and vt.nic=vas.nic) as t where \n" +
                                                            "c.gateid=g.gateid and g.orgid=o.orgid and c.orgid=o.orgid and t.cameraid=c.cameraid";

    private static final String SQL_INSERT_DEVICE_LOG = "insert into d_device_log(mall_id,mall_name,gate_id,gate_name,device_serialnum,device_ip,operate,operate_type,operate_date,operate_time) values(?,?,?,?,?,?,?,?,?,?)";

    private static final String SQL_GET_MALL_MAP = "SELECT id,unid from b_mall";

    private static final String SQL_GET_GATE_MAP = "select id,unid from b_gate";

    private static final String SQL_GET_MALL_LIST = "select id,name from b_mall";
    private static final String SQL_GET_GATE_LIST = "select id,name,is_mall_gate as mallGate from b_gate where mall_id=? order BY is_mall_gate desc";


    public List<DeviceDetails> getDeviceDetails() {
        if (version == 60) {
            return jdbcTemplate.query(SQL_GET_DEVICE_DETAILS, new BeanPropertyRowMapper<>(DeviceDetails.class));
        } else {
            return jdbcTemplate.query(SQL_GET_DEVICE_DETAILS_53, new BeanPropertyRowMapper<>(DeviceDetails.class));
        }
    }

    public Map<String, Long> getMallMap() {
        Map<String, Long> mallMap = new HashMap<>();
        jdbcTemplate.query(SQL_GET_MALL_MAP, rs -> {
            mallMap.put(rs.getString("unid"), rs.getLong("id"));
        });
        return mallMap;
    }

    public Map<String, Long> getGateMap() {
        Map<String, Long> gateMap = new HashMap<>();
        jdbcTemplate.query(SQL_GET_GATE_MAP, rs -> {
            gateMap.put(rs.getString("unid"), rs.getLong("id"));
        });
        return gateMap;
    }

    public void insertLogs(DeviceDetails details) {
        List<Operate> operates = details.getOperateList();
        jdbcTemplate.batchUpdate(SQL_INSERT_DEVICE_LOG, new BatchPreparedStatementSetter() {
            @Override
            public void setValues(PreparedStatement ps, int i) {
                Operate operate = operates.get(i);
                try {
                    ps.setObject(1, details.getMallId());
                    ps.setString(2, details.getMallName());
                    ps.setObject(3, details.getGateId());
                    ps.setString(4, details.getGateName());
                    ps.setString(5, details.getDeviceSerialNum());
                    ps.setString(6, details.getDeviceIP());
                    ps.setString(7, operate.getOperate());
                    ps.setString(8, operate.getType());
                    ps.setDate(9, new Date(operate.getOperateDate().getTime()));
                    ps.setTimestamp(10, new Timestamp(operate.getOperateTime().getTime()));
                } catch (Exception e) {
                    log.info("数据有误" + details.getDeviceIP() + ":" + details.getDeviceSerialNum());
                }

            }

            @Override
            public int getBatchSize() {
                if (operates != null) {
                    return operates.size();
                }
                return 0;
            }
        });
    }

    public List<LogInfo> getLogInfo(Long mallId, Long gateId, String deviceSerialNum, java.util.Date date, String type, Integer page) {
        List<Object> params = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        sb.append("select mall_name as mallName,gate_name as gateName,device_serialnum as deviceSerialNum,device_ip as deviceIp,operate,operate_date as operateDate,operate_time as operateTime from d_device_log where 1=1 ");
        if (mallId != null) {
            sb.append("and mall_id=? ");
            params.add(mallId);
        }
        if (gateId != null) {
            sb.append("and gate_id=? ");
            params.add(gateId);
        }
        if (deviceSerialNum != null && !"".equals(deviceSerialNum)) {
            sb.append("and device_serialnum=? ");
            params.add(deviceSerialNum);
        }
        if (date != null) {
            sb.append("and operate_date=? ");
            params.add(date);
        }
        if (!StringUtils.isEmpty(type)) {
            sb.append("and operate_type=? ");
            params.add(type);
        }
        sb.append("order by operate_time desc limit ").append(20).append(" offset ").append(20 * (page - 1));

        String sql = sb.toString();
        log.info("sql:{}\nparams:{}", sql, params.toString());

        return jdbcTemplate.query(sql, params.toArray(), new BeanPropertyRowMapper<>(LogInfo.class));
    }

    public Long getLogCount(Long mallId, Long gateId, String deviceSerialNum, java.util.Date date, String type) {
        List<Object> params = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        sb.append("select count(*) from d_device_log where 1=1 ");
        if (mallId != null) {
            sb.append("and mall_id=? ");
            params.add(mallId);
        }
        if (gateId != null) {
            sb.append("and gate_id=? ");
            params.add(gateId);
        }
        if (deviceSerialNum != null && !"".equals(deviceSerialNum)) {
            sb.append("and device_serialnum=? ");
            params.add(deviceSerialNum);
        }
        if (date != null) {
            sb.append("and operate_date=? ");
            params.add(date);
        }
        if (!StringUtils.isEmpty(type)) {
            sb.append("and operate_type=? ");
            params.add(type);
        }

        String sql = sb.toString();
        log.info("sql:{}\nparams:{}", sql, params.toString());
        return jdbcTemplate.queryForObject(sql, params.toArray(), Long.class);
    }

    public List<Map<String, Object>> getMallList() {
        List<Map<String, Object>> list = new ArrayList<>();
        jdbcTemplate.query(SQL_GET_MALL_LIST, rs -> {
            Map<String, Object> map = new HashMap<>();
            map.put("id", rs.getLong("id"));
            map.put("name", rs.getString("name"));
            list.add(map);
        });
        return list;
    }

    public List<Map<String, Object>> getGateList(Long mallId) {
        List<Map<String, Object>> list = new ArrayList<>();

        Map<String, Object> mallGateMap = new HashMap<>();
        List<Map<String, Object>> mallGateList = new ArrayList<>();
        mallGateMap.put("label", "带*的为外围监控点");
        mallGateMap.put("options", mallGateList);

        Map<String, Object> otherGateMap = new HashMap<>();
        List<Map<String, Object>> otherGateList = new ArrayList<>();
        otherGateMap.put("label", "非外围监控点");
        otherGateMap.put("options", otherGateList);

        list.add(mallGateMap);
        list.add(otherGateMap);

        jdbcTemplate.query(SQL_GET_GATE_LIST, new Object[]{mallId}, rs -> {
            Map<String, Object> map = new HashMap<>();
            map.put("id", rs.getLong("id"));

            if (rs.getInt("mallGate") > 0) {
                map.put("name", "*" + rs.getString("name"));
                mallGateList.add(map);
            } else {
                map.put("name", rs.getString("name"));
                otherGateList.add(map);
            }
        });

        return list;
    }
}
