package com.viontech.keliu.websocket;


import com.viontech.keliu.exception.ParameterExceptin;
import com.viontech.keliu.model.*;
import com.viontech.keliu.utils.HttpUtil;
import com.viontech.keliu.utils.JsonUtil;
import com.viontech.keliu.utils.UUIDUtil;
import com.viontech.keliu.utils.Util;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.websocket.DeploymentException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;


public class AlgApiClient {

    public static final String VENDOR_VION = "vion";
    public static final String VENDOR_BAIDU = "baidu";
    public static final String IMAGE_FORMAT_YUV_420I = "YUV_420I";
    public static final String IMAGE_FORMAT_YUV_420SPUV = "YUV_420SPUV";
    public static final String IMAGE_FORMAT_JPG = "jpg";
    public static final String IMAGE_TYPE_FACE = "face";
    public static final String IMAGE_TYPE_BODY = "body";
    public static final long API_REQUEST_TIME_OUT = (long) (1000 * 30);
    public String urlInfo = "";
    public static Map<String, Consumer<JSONObject>> rqid_consumer = new HashMap<>();
    //http请求标记，true为http请求，false为web socket请求。
    public Boolean httpFlag = false;
    Logger log = LoggerFactory.getLogger(AlgApiClient.class);
    private WsClientFacetory wsClientFacetory;
    private HttpUtil httpUtil;

    public AlgApiClient(String url) {
        if (url != null && url.contains("http")) {
            httpFlag = true;
            urlInfo = url;
            httpUtil = new HttpUtil();
        } else {
            httpFlag = false;
            wsClientFacetory = new WsClientFacetory();
            wsClientFacetory.setUrl(url);
        }
    }

    private Map<String, Object> getParams(String command, Consumer<JSONObject> callBack) {
        String rid = UUIDUtil.getUUID32();
        rqid_consumer.put(rid, callBack);
        Map<String, Object> params = new HashMap<>();
        params.put("command", command);
        params.put("rid", rid);
        return params;
    }

    /**
     * 2. 特征和属性提取
     */
    public CompletableFuture<JSONObject> getFeatureAndAttr(String image, String imageType, String imageFormat, List<FaceKey> faceKeys, Map<String, Object> options)
            throws ParameterExceptin, IOException, DeploymentException {


        if (image == null || "".equals(image)) {
            throw new ParameterExceptin("字段【image】不能为空");
        }
        if (imageType == null || "".equals(imageType)) {
            throw new ParameterExceptin("字段【imageType】不能为空");
        }
        if (imageFormat == null || "".equals(imageFormat)) {
            throw new ParameterExceptin("字段【imageFormat】不能为空");
        }
        if (IMAGE_TYPE_BODY.equals(imageType) && (options == null || options.get("body_roi") == null)) {
            throw new ParameterExceptin("当ImageType为face时，options中必须包含属性【body_roi】");
        }

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetFeatureAndAttr", callBack);

        if (!IMAGE_FORMAT_JPG.equals(imageFormat)) {
            Object image_width = options.get("image_width");
            if (image_width == null) {
                throw new ParameterExceptin("字段【image_width】不能为空");
            }

            Object image_height = options.get("image_height");
            if (image_height == null) {
                throw new ParameterExceptin("字段【image_height】不能为空");
            }
            Map definition = new HashMap();
            definition.put("width", image_width);
            definition.put("height", image_height);
            params.put("definition", definition);
        }
        String vendor = VENDOR_VION;
        if (options != null && options.get("vendor") != null) {
            vendor = String.valueOf(options.get("vendor"));
        }

        if (options != null && options.get("body_roi") != null) {
            ROI roi = (ROI) options.get("body_roi");
            params.put("bodyROI", roi);
        }

        if (options != null && options.get("needFaceCheck") != null) {
            Integer integer = (Integer) options.get("needFaceCheck");
            if (integer == 1) {//1为抠图
                params.put("needFaceCheck", integer);
            }
        }

        params.put("vendor", vendor);
        params.put("image", image);
        params.put("imageType", imageType);
        params.put("imageFormat", imageFormat);
        if (faceKeys != null) {
            float[] faceKeyArr = new float[faceKeys.size() * 2];
            int index = 0;
            for (int i = 0; i < faceKeys.size(); i++) {
                faceKeyArr[index++] = faceKeys.get(i).getX();
                faceKeyArr[index++] = faceKeys.get(i).getY();
            }
            params.put("faceKeys", faceKeyArr);
        }
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    /**
     * @param images       base64编码的图片的数组
     * @param imageFormat  图片格式，目前只支持"YUV_420I","YUV_420SPUV","jpg"这三种格式
     * @param faceKeyLists 人脸关键点的数组的数组
     * @param optionsList  options的数组
     *
     * @return void
     * @createDate 2018-11-13
     * @description 一次抓拍多张图片特征提取，参数列表中images，faceKeyLists和optionsList的大小必须一致
     */
    public CompletableFuture<JSONObject> getFaceFeatureBatch(List<String> images, String imageFormat, List<List<FaceKey>> faceKeyLists, List<Map<String, Object>> optionsList)
            throws ParameterExceptin, IOException, DeploymentException {


        if (Util.isEmpty(imageFormat)) {
            throw new ParameterExceptin("字段【imageFormat】不能为空");
        }
        if (images.size() <= 0) {
            throw new ParameterExceptin("参数长度不正确");
        }
        // 是否需要传分辨率
        boolean definitionFlag = false;
        if (!IMAGE_FORMAT_JPG.equals(imageFormat)) {
            definitionFlag = true;
            if (images.size() != optionsList.size()) {
                throw new ParameterExceptin("参数长度不正确");
            }
        }

        // 拼接faceImages
        List<Map<String, Object>> faceImages = new ArrayList<>();
        for (int i = 0; i < images.size(); i++) {
            // faceImage 封装一个face图片对象
            Map<String, Object> faceImage = new HashMap<>(4);
            // 对image 进行校验和封装
            if (Util.isEmpty(images.get(i))) {
                throw new ParameterExceptin("字段【image】不能为空");
            } else {
                faceImage.put("image", images.get(i));
            }
            // 封装imageFormat
            faceImage.put("imageFormat", imageFormat);
            // 判断是否封装definition
            if (definitionFlag) {
                Object imageWidth = optionsList.get(i).get("image_width");
                Object imageHeight = optionsList.get(i).get("image_height");
                if (imageWidth == null || imageHeight == null) {
                    throw new ParameterExceptin("字段【image_width】或【image_height】不能为空");
                }
                HashMap<Object, Object> definition = new HashMap<>(2);
                definition.put("width", imageWidth);
                definition.put("height", imageHeight);
                faceImage.put("definition", definition);
            }

            // 封装faceKeys
            if (faceKeyLists != null && faceKeyLists.size() > 0 && faceKeyLists.get(i) != null) {
                float[] faceKeyArr = new float[faceKeyLists.get(i).size() * 2];
                int index = 0;
                for (int j = 0; j < faceKeyLists.get(i).size(); j++) {
                    faceKeyArr[index++] = faceKeyLists.get(i).get(j).getX();
                    faceKeyArr[index++] = faceKeyLists.get(i).get(j).getY();
                }
                faceImage.put("faceKeys", faceKeyArr);
            }
            // 将封装好的人脸图片封装到列表中
            faceImages.add(faceImage);
        }

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetFaceFeatureBatch", callBack);
        // 拼接需要发送的数据
        params.put("vendor", VENDOR_VION);
        params.put("faceImages", faceImages);

        // 发送请求
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 3.1. 获取进店时间
     *
     * @param bodyFeatures
     * @param outTime
     * @param personPool
     */
    public CompletableFuture<JSONObject> getInTime(List<BodyFeature> bodyFeatures, String outTime, String personPoolId, List<Person> personPool, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (bodyFeatures == null || bodyFeatures.size() == 0) {
            throw new ParameterExceptin("字段【bodyFeatures】不能为空");
        }
        if (outTime == null || outTime == "") {
            throw new ParameterExceptin("字段【outTime】不能为空");
        }
        if (personPool == null) {
            personPool = new ArrayList<>();
        }

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetInTime", callBack);
        params.put("bodyFeatures", bodyFeatures);
        params.put("outTime", outTime);
        params.put("personPool", personPool);
        params.put("personPoolId", personPoolId);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 3.2. 获取异常滞留的顾客
     *
     * @param curTime
     * @param personPool
     */
    public CompletableFuture<JSONObject> getAbnormalPerson(String curTime, String personPoolId, List<Person> personPool, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (curTime == null || curTime == "")
            throw new ParameterExceptin("字段【curTime】不能为空");
        if (personPoolId == null || "".equals(personPoolId)) {
            throw new ParameterExceptin("字段【personPoolId】不能为空");
        }
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetAbnormalPerson", callBack);
        params.put("curTime", curTime);
        params.put("personPool", personPool);
        params.put("personPoolId", personPoolId);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.1.1. 人员特征池增加
     *
     * @param poolId
     * @param poolType
     * @param personPool
     */
    public CompletableFuture<JSONObject> addPersonPool(String poolId, int poolType, List<Person> personPool, Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {

        if (poolId == null || poolId == "")
            throw new ParameterExceptin("字段【poolId】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("AddPersonPool", callBack);
        params.put("poolId", poolId);
        params.put("poolType", poolType);
        if (personPool != null || personPool.size() > 0)
            params.put("personPool", personPool);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.1.2. 人员特征池删除
     *
     * @param flushPool
     * @param poolId
     * @param personIds
     */
    public CompletableFuture<JSONObject> delPersonPool(int flushPool, String poolId, List<Person> personIds, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (poolId == null || poolId == "")
            throw new ParameterExceptin("字段【poolId】不能为空");

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("DelPersonPool", callBack);
        params.put("flushPool", flushPool);
        params.put("poolId", poolId);
        params.put("personIds", personIds);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.1.4. 人员特征池查询
     *
     * @param listAll
     * @param poolId
     */
    public CompletableFuture<JSONObject> queryPersonPool(int listAll, String poolId, Map<String, Object> options)
            throws ParameterExceptin, IOException, DeploymentException {
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("QueryPersonPool", callBack);
        params.put("listAll", listAll);
        params.put("poolId", poolId);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.1.3. 人员特征池修改
     *
     * @param poolId
     * @param poolType
     * @param updateType
     * @param personPool
     */
    public CompletableFuture<JSONObject> modifyPersonPool(String poolId, int poolType, int updateType, List<Person> personPool, Map<String, Object> options)
            throws ParameterExceptin, IOException, DeploymentException {

        if (poolId == null || poolId == "")
            throw new ParameterExceptin("字段【poolId】不能为空");
        if (personPool == null || personPool.size() == 0)
            throw new ParameterExceptin("字段【personPool】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("ModifyPersonPool", callBack);
        params.put("poolId", poolId);
        params.put("poolType", poolType);
        params.put("updateType", updateType);
        params.put("personPool", personPool);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.2.1. 人员匹配
     *
     * @param personType
     * @param person
     * @param personPoolId
     * @param personPool
     */
    public CompletableFuture<JSONObject> matchPerson(int personType, Person person, String personPoolId, List<Person> personPool, Map<String, Object> options) throws ParameterExceptin, IOException, DeploymentException {

        if (person == null)
            throw new ParameterExceptin("字段【person】不能为空");
        if (personPoolId == null || personPoolId == "")
            throw new ParameterExceptin("字段【personPoolId】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("MatchPerson", callBack);
        params.put("personType", personType);
        params.put("person", person);
        params.put("personPoolId", personPoolId);
        params.put("personPool", personPool);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /***
     * 4.2.1. 新老顾客匹配
     * @param personType
     * @param person
     * @param unionPersonPoolId
     * @param options
     * @throws ParameterExceptin
     * @throws IOException
     * @throws DeploymentException
     */
    public CompletableFuture<JSONObject> matchHistoryPerson(int personType, Person person, List<String> unionPersonPoolId, Map<String, Object> options)
            throws ParameterExceptin, IOException, DeploymentException {

        if (person == null)
            throw new ParameterExceptin("字段【person】不能为空");
        if (unionPersonPoolId == null || unionPersonPoolId.size() <= 0)
            throw new ParameterExceptin("字段【unionPersonPoolId】不能为空且不能没有池名称");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("MatchPerson", callBack);
        params.put("personType", personType);
        params.put("person", person);
        params.put("unionPersonPoolId", unionPersonPoolId);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.3. 人员更新
     *
     * @param personType
     * @param person
     * @param newFaceFeature
     */
    public CompletableFuture<JSONObject> updatePerson(int personType, Person person, FaceFeature newFaceFeature, Map<String, Object> options) throws ParameterExceptin, IOException, DeploymentException {

        if (person == null)
            throw new ParameterExceptin("字段【person】不能为空");
        if (newFaceFeature == null)
            throw new ParameterExceptin("字段【newFaceFeature】不能为空");

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;

        Map<String, Object> params = getParams("UpdatePerson", callBack);
        params.put("personType", personType);
        params.put("person", person);
        params.put("newFaceFeature", newFaceFeature);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.4. 人员校验
     *
     * @param personType
     * @param person
     */
    public CompletableFuture<JSONObject> verifyPerson(int personType, Person person, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (person == null)
            throw new ParameterExceptin("字段【person】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("VerifyPerson", callBack);
        params.put("personType", personType);
        params.put("person", person);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.5. 人员库聚类
     *
     * @param personType
     * @param personPoolId
     * @param personPool
     */
    public CompletableFuture<JSONObject> clusterPerson(int personType, String personPoolId, List<Person> personPool, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (personPoolId == null || personPoolId == "")
            throw new ParameterExceptin("字段【personPoolId】不能为空");
        if (personPool == null || personPool.size() == 0)
            throw new ParameterExceptin("字段【personPool】不能为空");

        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;

        Map<String, Object> params = getParams("ClusterPerson", callBack);
        params.put("personType", personType);
        params.put("personPoolId", personPoolId);
        params.put("personPool", personPool);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 4.6. 人员库校验
     *
     * @param personType
     * @param personPoolId
     * @param personPool
     */
    public CompletableFuture<JSONObject> verifyPersonPool(int personType, String personPoolId, List<Person> personPool, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {

        if (personPoolId == null || personPoolId == "")
            throw new ParameterExceptin("字段【personPoolId】不能为空");
        if (personPool == null || personPool.size() == 0)
            throw new ParameterExceptin("字段【personPool】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("VerifyPersonPool", callBack);
        params.put("personType", personType);
        params.put("personPoolId", personPoolId);
        params.put("personPool", personPool);

        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 5. 动线统计
     *
     * @param motionPoints
     */
    public CompletableFuture<JSONObject> getMotionTrack(List<Person> motionPoints, Map<String, Object> options)
            throws IOException, DeploymentException, ParameterExceptin {
        if (motionPoints == null || motionPoints.size() == 0)
            throw new ParameterExceptin("字段【motionPoints】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetMotionTrack", callBack);
        params.put("motionPoints", motionPoints);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    /**
     * 6.1 输入人员信息
     */
    public CompletableFuture<JSONObject> addCaptureRecords(List<CaptureRecord> captureRecords, Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object calId = options.get("calId");
        if (calId == null) {
            throw new ParameterExceptin("字段【calId】不能为空");
        }
        Object clerkPoolId = options.get("clerkPoolId");
        if (clerkPoolId == null) {
            throw new ParameterExceptin("字段【clerkPoolId】不能为空");
        }
        Object takeawayPersonPoolId = options.get("takeawayPersonPoolId");
        if (takeawayPersonPoolId == null) {
            throw new ParameterExceptin("字段【takeawayPersonPoolId】不能为空");
        }
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("AddCaptureRecords", callBack);
        params.put("calId", calId);
        params.put("clerkPoolId", clerkPoolId);
        params.put("takeawayPersonPoolId", takeawayPersonPoolId);
        if (captureRecords != null || captureRecords.size() > 0)
            params.put("captureRecords", captureRecords);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 6.1.2 查询人员
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> queryCaptureRecords(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object calId = options.get("calId");
        if (calId == null) throw new ParameterExceptin("字段【calId】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("QueryCaptureRecords", callBack);
        params.put("calId", calId);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    /**
     * 6.1.3 删除人员
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> deleteCaptureRecords(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object calId = options.get("calId");
        if (calId == null) throw new ParameterExceptin("字段【calId】不能为空");
//        Object clearAll = options.get("clearAll");
//        if (clearAll == null) throw new ParameterExceptin("字段【clearAll】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("DeleteCaptureRecords", callBack);
        params.put("calId", calId);
        params.put("clearAll", 1);//全部删除
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 6.1.4 查询人员池
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> queryCalIds(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("QueryCalIds", callBack);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 6.2.1 获取分析结果概述
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> analyzeRecords(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object calId = options.get("calId");
        if (calId == null) throw new ParameterExceptin("字段【calId】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("AnalyzeRecords", callBack);
        params.put("calId", calId);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }

    /**
     * 6.2.1 获取分析结果概述
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> analyzeRecords(Map<String, Object> options, Long timeout) throws IOException, DeploymentException, ParameterExceptin {
        Object calId = options.get("calId");
        if (calId == null) throw new ParameterExceptin("字段【calId】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("AnalyzeRecords", callBack);
        params.put("calId", calId);
        httpOrWebSocketSend(callBack, params, timeout);
        return completableFuture;
    }

    /**
     * 6.2.2 获取订单详情
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> GetOrderDetails(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object transactionId = options.get("transactionId");
        Object beginIndex = options.get("beginIndex");
        Object endIndex = options.get("endIndex");
        if (transactionId == null) throw new ParameterExceptin("字段【transactionId】不能为空");
        if (beginIndex == null) throw new ParameterExceptin("字段【beginIndex】不能为空");
        if (endIndex == null) throw new ParameterExceptin("字段【endIndex】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetOrderDetails", callBack);
        params.put("transactionId", transactionId);
        params.put("beginIndex", beginIndex);
        params.put("endIndex", endIndex);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    /**
     * 6.2.3. 获取需要纠正的人员信息详情
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> getVerifiedPersonDetails(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object transactionId = options.get("transactionId");
        Object beginIndex = options.get("beginIndex");
        Object endIndex = options.get("endIndex");
        if (transactionId == null) throw new ParameterExceptin("字段【transactionId】不能为空");
        if (beginIndex == null) throw new ParameterExceptin("字段【beginIndex】不能为空");
        if (endIndex == null) throw new ParameterExceptin("字段【endIndex】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetVerifiedPersonDetails", callBack);
        params.put("transactionId", transactionId);
        params.put("beginIndex", beginIndex);
        params.put("endIndex", endIndex);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    /**
     * 6.2.4. 获取标准特征详情
     *
     * @param options
     *
     * @return
     */
    public CompletableFuture<JSONObject> getStandardPersonDetails(Map<String, Object> options) throws IOException, DeploymentException, ParameterExceptin {
        Object transactionId = options.get("transactionId");
        Object beginIndex = options.get("beginIndex");
        Object endIndex = options.get("endIndex");
        if (transactionId == null) throw new ParameterExceptin("字段【transactionId】不能为空");
        if (beginIndex == null) throw new ParameterExceptin("字段【beginIndex】不能为空");
        if (endIndex == null) throw new ParameterExceptin("字段【endIndex】不能为空");
        CompletableFuture<JSONObject> completableFuture = new CompletableFuture<>();
        Consumer<JSONObject> callBack = completableFuture::complete;
        Map<String, Object> params = getParams("GetStandardPersonDetails", callBack);
        params.put("transactionId", transactionId);
        params.put("beginIndex", beginIndex);
        params.put("endIndex", endIndex);
        httpOrWebSocketSend(callBack, params);
        return completableFuture;
    }


    private void httpOrWebSocketSend(Consumer<JSONObject> callBack, Map<String, Object> params) throws IOException, DeploymentException {
        if (httpFlag) {
            HashMap hashMap = (HashMap) params;
            httpUtil.send(urlInfo, hashMap, callBack);
        } else {
            wsClientFacetory.send(JsonUtil.toStr(params));
        }
    }

    private void httpOrWebSocketSend(Consumer<JSONObject> callBack, Map<String, Object> params, Long timeout) throws IOException, DeploymentException {
        if (httpFlag) {
            HashMap hashMap = (HashMap) params;
            httpUtil.send(urlInfo, hashMap, callBack);
        } else {
            wsClientFacetory.send(JsonUtil.toStr(params), timeout);
        }
    }
}
